/*
 *  ServerLink.cpp
 *  
 *
 *  Created by Jason Toffaletti on Sat Apr 26 2003.
 *  Copyright (c) 2003 __MyCompanyName__. All rights reserved.
 *
 */
#ifdef CONFIG_LINKING
#include "ServerLink.h"
#include "HLServer.h"
#include "HLProtocol.h"
#include "HLClient.h"

struct flattenable_user : public flattenable
{
	virtual ~flattenable_user() {} // to get rid of gcc warning - jjt
	
	flattenable_string name;
	flattenable_uint32 icon;
	flattenable_uint32 id;
	
	const flat_type_code type_code() const
		{ return FLAT_INVALID_TYPE; } // not sure what to do here yet - jjt

	const bool is_fixed_size() const { return false; }
	
	const unsigned int flattened_size() const
	{
		return (name.flattened_size() + icon.flattened_size() + id.flattened_size());
	}
	
	void flatten(std::ostream &strm) const
	{
		name.flatten(strm);
		icon.flatten(strm);
		id.flatten(strm);
	}
	
	void unflatten(std::istream &strm)
	{
		name.unflatten(strm);
		icon.unflatten(strm);
		id.unflatten(strm);
	}
};

typedef flattenable_list<flattenable_user> flattenable_user_list;

LinkSocket::LinkSocket(LinkSocketCallback *inCallback)
	: BufferedSocket(), mCallback(inCallback)
{
}

void LinkSocket::DataAvailable()
{
	if (mReadBuf.Size())
	{
		DEBUG_CALL(printf("read buf size: %lu\n", mReadBuf.Size()));
		string dataBuf(mReadBuf.Ptr(), mReadBuf.Size());
		// here i append a character to the end of the string
		// as a really nasty way of preventing an exception from being thrown
		// when the end of the stream is reached. it makes no fucking sense
		// why reaching the end of the stream is an error, but it is...
		dataBuf.append(" ");
		istringstream strm(dataBuf);
		strm.exceptions(ios::eofbit | ios::failbit | ios::badbit);
		while (!strm.eof())
		{
			try
			{
				jmessage msg;
				msg.unflatten(strm);
				mReadBuf.RemoveFrom(msg.flattened_size());
				if (mCallback)
					mCallback->ProcessJMessage(*this, msg);
			}
			catch (ios_base::failure &ioerr)
			{
				if (!strm.eof())
					DEBUG_CALL(printf("data avail io failure: %s\n", ioerr.what()));
			}
		}
	}
}

void LinkSocket::OnConnect()
{
	if (mCallback)
		mCallback->HandleConnect();
}

void LinkSocket::OnClose()
{
	if (mCallback)
		mCallback->HandleClose();
}

void LinkSocket::SendJMessage(jmessage &inMsg)
{
	if (IsConnected())
	{
		ostringstream strm;
		strm.exceptions(ios::eofbit | ios::failbit | ios::badbit);
		try
		{
			inMsg.flatten(strm);
			DEBUG_CALL(printf("sending jmessage %lu bytes\n", strm.str().size()));
			BufferedSend(strm.str().data(), strm.str().size());
		}
		catch (ios_base::failure &ioerr)
		{
			DEBUG_CALL(printf("snd jmsg io failure: %s\n", ioerr.what()));
		}
		catch (socket_error &exp)
		{
			DEBUG_CALL(printf("socket exception in LinkSocket::SendJMessage %s\n", exp.what()));
		}
	}
}

LinkManager::LinkManager()
	//: mLinkSocket(this)
{
	mLinkSocket = this;
}

void LinkManager::EstablishLinkWithHub(const string &inAddress, u_int16_t inPort)
{
	mLinkSocket.Open();
	mLinkSocket.SetSelector(gServer->mSelector);
	mLinkSocket.Connect(inAddress, inPort);
}

void LinkManager::LinkChat(HLClient &inClient, unsigned int inStyle, const string &inChat)
{
	// at some point i should add chatID and userID - jjt
	jmessage msg;
	msg.what = 'chat';
	msg.add_string(0, inChat);
	msg.add_uint32(1, inClient.User().GlobalID());
	msg.add_uint32(2, inStyle);
	mLinkSocket.SendJMessage(msg);
}

void LinkManager::LinkBroadcast(const string &inBroadcast)
{
	jmessage msg;
	msg.what = 'bcst';
	msg.add_string(0, inBroadcast);
	mLinkSocket.SendJMessage(msg);
}

void LinkManager::LinkMessage(HLClient &sendClient, HLClient &toClient, const string &inMessage)
{
	jmessage msg;
	msg.what = 'pmsg';
	msg.add_uint32(0, sendClient.User().GlobalID());
	msg.add_uint32(1, toClient.User().GlobalID());
	msg.add_string(2, inMessage);
	mLinkSocket.SendJMessage(msg);
}

void LinkManager::LinkUserJoin(HLClient &inClient)
{
	jmessage msg;
	msg.what = 'uadd';
	msg.add_string(0, inClient.User().Name());
	msg.add_uint32(1, inClient.User().ID());
	msg.add_uint32(2, inClient.User().Icon());
	msg.add_uint32(3, inClient.User().Status());
	msg.add_uint32(4, inClient.User().LoginTime());
	mLinkSocket.SendJMessage(msg);
}

void LinkManager::LinkUserChange(HLClient &inClient)
{
	jmessage msg;
	msg.what = 'uchg';
	msg.add_string(0, inClient.User().Name());
	msg.add_uint32(1, inClient.User().GlobalID());
	msg.add_uint32(2, inClient.User().Icon());
	msg.add_uint32(3, inClient.User().Status());
	msg.add_uint32(4, inClient.User().LoginTime());
	msg.add_uint32(5, inClient.User().LastRecvTime());
	mLinkSocket.SendJMessage(msg);
}

void LinkManager::LinkUserLeave(HLClient &inClient)
{
	jmessage msg;
	msg.what = 'ulve';
	// TODO: should change this to use global id once i implement the msg queue - jjt
	msg.add_uint32(0, inClient.User().ID());
	mLinkSocket.SendJMessage(msg);
}

void LinkManager::ProcessJMessage(LinkSocket &inSocket, jmessage &inMsg)
{
	StMutexLock lock(gServer->mLock);
	(void)inSocket;
	switch (inMsg.what)
	{
		case 'chat':
			HandleChat(inMsg);
			break;
		
		case 'bcst':
			HandleBroadcast(inMsg);
			break;
		
		case 'pmsg':
			HandleMessage(inMsg);
			break;
			
		case 'sgid':
			HandleSetGuid(inMsg);
			break;
		
		case 'uadd':
			HandleUserJoin(inMsg);
			break;
		
		case 'ulve':
			HandleUserLeave(inMsg);
			break;
		
		case 'uchg':
			HandleUserChange(inMsg);
			break;
			
		default:
			DEBUG_CALL(printf("unhandled jmsg what: %u\n", (unsigned int)inMsg.what));
			break;
	}
}

void LinkManager::HandleConnect()
{
	jmessage msg;
	msg.what = 'helo';
	flattenable_user_list userList;
	flattenable_user user;
	
	StMutexLock lock(gServer->mLock);
	ClientList::iterator iter = gServer->mClientList.begin();
	while (iter != gServer->mClientList.end())
	{
		if ((*iter)->User().IsLoggedIn() && (*iter)->User().LoginTime())
		{
			user.name = (*iter)->User().Name();
			user.icon = (*iter)->User().Icon();
			user.id = (*iter)->User().ID();
			userList.push_back(user);
		}
		iter++;
	}
	
	msg.add_object(0, userList);
	mLinkSocket.SendJMessage(msg);
}

void LinkManager::HandleClose()
{
	// this should do something like try to reconnect after a few seconds
	
	// it should also remove all remote users from the userlist
	gServer->RemoveRemoteClients();
}

void LinkManager::HandleChat(jmessage &inMsg)
{
	string chatString;
	uint32 globalID;
	uint32 style;
	inMsg.find_string(0, chatString);
	inMsg.find_uint32(1, globalID);
	inMsg.find_uint32(2, style);
	
	HLClient *chatClient = gServer->GetClientWithGlobalID(globalID);
	if (chatClient)
	{
		unsigned int lineCount;
		HLServer::FormatChat(*chatClient, style, chatString, lineCount);
		HLPacket chatPacket(HTLS_HDR_CHAT, 0, 0);
		chatPacket.AddStringObject(HTLS_DATA_CHAT, chatString);
		chatPacket.AddUInt16Object(HTLS_DATA_UID, chatClient->User().ID());
		gServer->SendToAll(chatPacket, HLServer::ReadChatFilter);
	}
}

void LinkManager::HandleBroadcast(jmessage &inMsg)
{
	string broadcastString;
	if (inMsg.find_string(0, broadcastString))
	{
		HLPacket broadcastPacket(HTLS_HDR_MSG_BROADCAST, 0, 0);
        broadcastPacket.AddStringObject(HTLS_DATA_MSG, broadcastString);
		gServer->SendToAll(broadcastPacket);
	}
}

void LinkManager::HandleMessage(jmessage &inMsg)
{
	uint32 sender_guid;
	uint32 recipient_guid;
	string msgString;
	inMsg.find_uint32(0, sender_guid);
	inMsg.find_uint32(1, recipient_guid);
	inMsg.find_string(2, msgString);
	
	HLClient *senderClient = gServer->GetClientWithGlobalID(sender_guid);
	HLClient *recipientClient = gServer->GetClientWithGlobalID(recipient_guid);
	if (senderClient && recipientClient)
	{
		HLPacket msgPacket(HTLS_HDR_MSG, 0, 0);
		msgPacket.AddUInt16Object(HTLS_DATA_UID, senderClient->User().ID());
		msgPacket.AddStringObject(HTLS_DATA_NAME, senderClient->User().Name());
		msgPacket.AddStringObject(HTLS_DATA_MSG, msgString);
		recipientClient->SendPacket(msgPacket);
	}
}

void LinkManager::HandleSetGuid(jmessage &inMsg)
{
	uint32 userID;
	uint32 globalID;
	inMsg.find_uint32(0, userID);
	inMsg.find_uint32(1, globalID);
	
	HLClient *theClient = gServer->GetClientForID(userID);
	if (theClient)
		theClient->User().SetGlobalID(globalID);
}

void LinkManager::HandleUserJoin(jmessage &inMsg)
{
	string userName;
	uint32 globalID;
	uint32 userIcon;
	uint32 userStatus;
	
	inMsg.find_string(0, userName);
	inMsg.find_uint32(1, globalID);
	inMsg.find_uint32(2, userIcon);
	inMsg.find_uint32(3, userStatus);
	
	DEBUG_CALL(printf("got remote user join: %u\n", (unsigned int)globalID));
	HLAccount theAccount;
	if (gServer->mDataBase.FetchAccount(theAccount, false))
	{
		new HLClient(userName, globalID, userIcon, userStatus, theAccount);
	}
}

void LinkManager::HandleUserLeave(jmessage &inMsg)
{
	uint32 globalID;
	inMsg.find_uint32(0, globalID);
	DEBUG_CALL(printf("got remote user leave: %u\n", (unsigned int)globalID));
	gServer->RemoveClientWithGlobalID(globalID);
}

void LinkManager::HandleUserChange(jmessage &inMsg)
{
	DEBUG_CALL(printf("got remote user change\n"));
	
	uint32 globalID;
	inMsg.find_uint32(1, globalID);
	HLClient *client = gServer->GetClientWithGlobalID(globalID);
	if (client)
	{
		string name;
		uint32 icon;
		inMsg.find_string(0, name);
		inMsg.find_uint32(2, icon);
	
		client->User().SetName(name);
		client->User().SetIcon(icon);
		// client is obviously not idle anymore
		// TODO: i need to rewrite all this idle stuff later to be more like
		// what i do in HLClient once the protocol is set
		client->User().SetStatus(client->User().Status() ^ IDLE_MASK);
		
		HLPacket userChangePacket(HTLS_HDR_USER_CHANGE, 0, 0);
		userChangePacket.AddStringObject(HTLS_DATA_NAME, client->User().Name());
		userChangePacket.AddUInt16Object(HTLS_DATA_UID, client->User().ID());
		userChangePacket.AddUInt16Object(HTLS_DATA_ICON, client->User().Icon());
		userChangePacket.AddUInt16Object(HTLS_DATA_COLOUR, client->User().Status());
		
		gServer->SendToAll(userChangePacket);
	}
}
#endif//CONFIG_LINKING //2003/11/03 added by ortana.
