/*
 *  This file is part of Terra Hotline Server.
 *
 *	Terra is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License.
 *
 *	Terra is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with Terra; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// Where it all begins...

#include "HLServer.h"
#include <limits.h>
#include "FileUtils.h"
#include "ServerLog.h"

#include <fcntl.h>

#if !defined(WIN32)
#include <signal.h>
#include <unistd.h>
#include <errno.h>
static void sig_handler(int sig);
#endif // !WIN32

static void cleanup(void);

#if defined(WIN32)

// Definitions for getopt
extern "C" {
extern char *optarg;
extern int optreset;
extern int optind;
extern int opterr;
extern int optopt;
int getopt( int argc, char* const *argv, const char *optstr );
}

static bool InitWinsock()
{
	WORD wVersionRequested;
	WSADATA wsaData;
	int err;
 
	wVersionRequested = MAKEWORD( 2, 2 );
 
	err = WSAStartup( wVersionRequested, &wsaData );
	if ( err != 0 ) {
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		return false;
	}
 
	/* Confirm that the WinSock DLL supports 2.2.*/
	/* Note that if the DLL supports versions greater    */
	/* than 2.2 in addition to 2.2, it will still return */
	/* 2.2 in wVersion since that is the version we      */
	/* requested.                                        */
 
	if ( LOBYTE( wsaData.wVersion ) != 2 ||
			HIBYTE( wsaData.wVersion ) != 2 ) {
		/* Tell the user that we could not find a usable */
		/* WinSock DLL.                                  */
		WSACleanup( );
		return false; 
	};

	return true;
}



#endif //WIN32

int main (int argc, char * const argv [])
{
	int ch;
	ServerConf theConf;
    bool commandLineConf = false;
    u_int16_t serverPort = 0;
    bool savePid = false;
    string pidPath;
	uid_t userID = 0;
	gid_t groupID = 0;
    
#ifdef WIN32
	if( !InitWinsock() )
	{
		cerr << "Unable to load winsock" << endl;
		return 1;
	}
#endif // WIN32

    if (atexit(cleanup) == -1)
    {
        ServerLog::ErrorLog(__FILE__, __LINE__, "atexit() returned: %s", strerror(errno));
        return -1;
    }
    
    try
    {
        while ((ch = getopt(argc, argv, "r:p:f:")) != -1)
        {
            switch (ch)
            {
                case 'r':
                    savePid = true;
                    pidPath = optarg;
                    break;
                    
                case 'p':
                    // save this for later, to override any conf file settings
                    serverPort = strtoul(optarg, 0, 10);
                    break;
                
                case 'f':
                    commandLineConf = true;
                    theConf.ReadConf(optarg);
                    break;
				
				case 'u':
					// save this for later, to override any conf file settings
                    userID = strtoul(optarg, 0, 10);
					break;
				
				case 'g':
					// save this for later, to override any conf file settings
                    groupID = strtoul(optarg, 0, 10);
					break;
                
                case '?':
                default:
                    printf("usage: [-r pid path][-f conf file][-p port][-u uid][-g gid]\n");
                    return 0;
            }
        }
		
		if (!commandLineConf)
     {
            // i'm not sure if this is the way the search path should work
            // this is mantis bug id 15
            // this seems a bit backwards to me...need to ask sam about his bug report
            if (!theConf.ReadConf("/etc/terra/terra.conf"))
            {
                if (!theConf.ReadConf("~/.terra/terra.conf"))
                {
									//2003/07/30 modified by ortana.
#ifdef WIN32
									theConf.ReadConf("data\\winterra.xml");
#else
                    theConf.ReadConf("terra.conf");
#endif
                }
            }
        }
    }
    catch (exception &exp)
    {
        ServerLog::ErrorLog(__FILE__, __LINE__, "fatal exception %s", exp.what());
        return -1;
    }
	
	if (userID != 0)
		theConf.userID = userID;
	
	if (groupID != 0)
		theConf.groupID = groupID;
    
    if (serverPort != 0)
        theConf.serverPort = serverPort;

#if !defined(WIN32)
	// I'm not sure if Windows supports these calls, but probably not
	// I also am not sure if these errors setting uid and gid should
	// be fatal or only logged, hard to decide.
	int status;
	if (theConf.userID != 0)
	{
		status = setuid(theConf.userID);
		if (status == -1)
			ServerLog::ErrorLog(__FILE__, __LINE__, "setuid() returned: %s", strerror(errno));
	}
	
	if (theConf.groupID != 0)
	{
		status = setgid(theConf.groupID);
		if (status == -1)
			ServerLog::ErrorLog(__FILE__, __LINE__, "setgid() returned: %s", strerror(errno));
    }
	
	// NOTE: Windows doesn't support these unix signals
    // i still need to ignore SIGPIPE because some systems don't
    // support disabling SIGPIPE for the EPIPE error
    signal(SIGPIPE, SIG_IGN);
    
    // for quiting the server
    signal(SIGINT, sig_handler);
    signal(SIGQUIT, sig_handler);
    signal(SIGKILL, sig_handler);
    
    // for reloading the config file
    signal(SIGHUP, sig_handler);

#if !defined(_DEBUG_BUILD_)
    // this turns the server into a background only daemon
    // i need to have error log go to a file or use syslog()?
    if (daemon(1, 0) == -1)
    {
        ServerLog::ErrorLog(__FILE__, __LINE__, "daemon() returned: %s", strerror(errno));
        return -1;
    }
#endif
    
    if (savePid)
    {
        int fd;
        fd = open(pidPath.c_str(), O_WRONLY | O_CREAT, 0644);
        if (fd >= 0)
        {
            char pidStr[64];
            snprintf(pidStr, 63, "%u", getpid());
            write(fd, pidStr, strlen(pidStr));
        }
        close(fd);
    }
#endif // !WIN32

    HLServer hlServer(theConf);
    try
    {
        FileUtils::createTypeMaps();
        gServer->Start();
    }
    catch (exception &exp)
    {
        ServerLog::ErrorLog(__FILE__, __LINE__, "exception caught in main: %s", exp.what());
    }
    
    // removing this allows server to exit without a segfault
    // i think this is because other threads haven't exited
    // and they are still trying to reference gServer
    //DEBUG_CALL(printf("deleting server\n"); fflush(stdout));
    //delete gServer;
    if (savePid)
        FileUtils::deletePath(pidPath);
    
	return 0;
}


static void cleanup(void)
{
#ifdef WIN32
	WSACleanup();
#endif

    DEBUG_CALL(printf("my atexit\n"); fflush(stdout));
}

#if !defined(WIN32)
static void sig_handler(int sig)
{
	switch (sig)
	{
		case SIGINT:
		case SIGQUIT:
		case SIGKILL:
			// this is cleanup
			if (gServer && gServer->IsRunning())
				gServer->Stop();
			break;
		
		case SIGHUP:
			if (gServer)
				gServer->ReloadConf();
			break;
        
		default:
			DEBUG_CALL(printf("unhandled sig: %d\n", sig));
			break;
	}
}
#endif // !WIN32
