/*
 *  types.hpp
 *  serialize
 *
 *  Created by Jason Toffaletti on Fri Apr 18 2003.
 *  Copyright (c) 2003 __MyCompanyName__. All rights reserved.
 *
 */

#ifndef FLATTYPES_H
#define FLATTYPES_H

#include <string>
#include <list>
#include <vector>
#include <map>
#include <boost/scoped_array.hpp>
#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include "flattenable.hpp"

#ifdef WIN32
typedef signed char sint8;
typedef unsigned char uint8;
typedef signed short sint16;
typedef unsigned short uint16;
typedef signed long sint32;
typedef unsigned long uint32;
typedef __int64 sint64;
typedef unsigned __int64 uint64;

#pragma warning( disable : 4800 )//2003/11/03 added by ortana.

#else
typedef signed char sint8;
typedef unsigned char uint8;
typedef signed short sint16;
typedef unsigned short uint16;
typedef signed long sint32;
typedef unsigned long uint32;
typedef signed long long sint64;
typedef unsigned long long uint64;
#endif

#define	FLAT_BOOL_TYPE		(flat::flat_type_code)'bool'
#define	FLAT_SINT8_TYPE		(flat::flat_type_code)'sit8'
#define	FLAT_UINT8_TYPE		(flat::flat_type_code)'uit8'
#define	FLAT_SINT16_TYPE	(flat::flat_type_code)'si16'
#define	FLAT_UINT16_TYPE	(flat::flat_type_code)'ui16'
#define	FLAT_SINT32_TYPE	(flat::flat_type_code)'si32'
#define	FLAT_UINT32_TYPE	(flat::flat_type_code)'ui32'
#define	FLAT_SINT64_TYPE	(flat::flat_type_code)'si64'
#define	FLAT_UINT64_TYPE	(flat::flat_type_code)'ui64'
#define FLAT_LIST_TYPE		(flat::flat_type_code)'list'
#define FLAT_VECTOR_TYPE	(flat::flat_type_code)'vect'
#define FLAT_MAP_TYPE		(flat::flat_type_code)'_map'

namespace flat {

template<typename T> inline T muscleSwapBytes(T swapMe)
{
   T retVal;
   const uint8 * readFrom = (const uint8 *) &swapMe;
   uint8 * writeTo  = ((uint8 *) &retVal)+sizeof(retVal);
   for (uint32 i=0; i<sizeof(swapMe); i++) {*(--writeTo) = *(readFrom++);}
   return retVal;
}

class flattenable_sint8 : public basic_flattenable<sint8>
{
public:
	flattenable_sint8() { m_data = 0; }
	flattenable_sint8(const sint8 data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_SINT8_TYPE; }
};

class flattenable_uint8 : public basic_flattenable<uint8>
{
public:
	flattenable_uint8() { m_data = 0; }
	flattenable_uint8(const uint8 data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_UINT8_TYPE; }
};

class flattenable_bool : public basic_flattenable<bool>
{
public:
	flattenable_bool() { m_data = false; }
	flattenable_bool(const bool data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_BOOL_TYPE; }
	const bool is_fixed_size() const { return true; }
	const unsigned int flattened_size() const { return sizeof(uint8); }
	
	void flatten(std::ostream &strm) const
	{
		// bool is usually an int but we only need to store one byte
		flattenable_uint8 tmp = m_data;
		tmp.flatten(strm);
	}
	
	void unflatten(std::istream &strm)
	{
		flattenable_uint8 tmp;
		tmp.unflatten(strm);
		m_data = tmp;
	}
};

class flattenable_sint16 : public basic_flattenable<sint16>
{
public:
	flattenable_sint16() { m_data = 0; }
	flattenable_sint16(const sint16 data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_SINT16_TYPE; }
	
	const sint16 hton(const sint16 t1) const { return htons(t1); }
	const sint16 ntoh(const sint16 t1) const { return ntohs(t1); }
};

class flattenable_uint16 : public basic_flattenable<uint16>
{
public:
	flattenable_uint16() { m_data = 0; }
	flattenable_uint16(const uint16 data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_UINT16_TYPE; }
	
	const uint16 hton(const uint16 t1) const { return htons(t1); }
	const uint16 ntoh(const uint16 t1) const { return ntohs(t1); }
};

class flattenable_sint32 : public basic_flattenable<sint32>
{
public:
	flattenable_sint32() { m_data = 0; }
	flattenable_sint32(const sint32 &data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_SINT32_TYPE; }
	
	const sint32 hton(const sint32 t1) const { return htonl(t1); }
	const sint32 ntoh(const sint32 t1) const { return ntohl(t1); }
};

class flattenable_uint32 : public basic_flattenable<uint32>
{
public:
	flattenable_uint32() { m_data = 0; }
	flattenable_uint32(const uint32 &data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_UINT32_TYPE; }
	
	const uint32 hton(const uint32 t1) const { return htonl(t1); }
	const uint32 ntoh(const uint32 t1) const { return ntohl(t1); }
};

class flattenable_sint64 : public basic_flattenable<sint64>
{
public:
	flattenable_sint64() { m_data = 0; }
	flattenable_sint64(const sint64 &data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_SINT64_TYPE; }
};

class flattenable_uint64 : public basic_flattenable<uint64>
{
public:
	flattenable_uint64() { m_data = 0; }
	flattenable_uint64(const uint64 &data) { m_data = data; }
	const flat_type_code type_code() const { return FLAT_UINT64_TYPE; }
};

#ifndef WIN32
// microsoft didn't put string in std namespace
using namespace std;
#endif

class flattenable_string : public flattenable, public string
{
public:
	flattenable_string() {}
	flattenable_string(const char *s) : string(s) {}
	flattenable_string(const string &l) : string(l) {}
	virtual ~flattenable_string() {}

	const flat_type_code type_code() const { return FLAT_LIST_TYPE; }
	const bool is_fixed_size() const { return false; }
	
	const unsigned int flattened_size() const
	{
		return (sizeof(uint32) + string::size());
	}
	
	void flatten(std::ostream &strm) const
	{
		flattenable_uint32 ssize = string::size();
		ssize.flatten(strm);
		strm.write(string::data(), string::size());
	}
	
	void unflatten(std::istream &strm)
	{
		flattenable_uint32 ssize;
		ssize.unflatten(strm);
		boost::scoped_array<char> buf(new char[ssize]);
		strm.read(buf.get(), ssize);
		string::assign(buf.get(), ssize);
	}
};

template <typename T>
class flattenable_list : public flattenable, public std::list<T>
{
public:
	typedef std::list<T> list_type;

	flattenable_list() {}
	flattenable_list(const list_type &l) : list_type(l) {}
	virtual ~flattenable_list() {}
	const flat_type_code type_code() const { return FLAT_LIST_TYPE; }
	const bool is_fixed_size() const { return false; }
	
	const unsigned int flattened_size() const
	{
		unsigned int flat_size = sizeof(uint32);
	
		if (!list_type::empty())
		{
			typename list_type::const_iterator iter = begin();
			if ((*iter).is_fixed_size())
				flat_size += (list_type::size() * (*iter).flattened_size());
			else
			{
				while (iter != end())
				{
					flat_size += (*iter).flattened_size();
					iter++;
				}
			}
		}
		return flat_size;
	}
	
	void flatten(std::ostream &strm) const
	{
		flattenable_uint32 nitems = list_type::size();
		nitems.flatten(strm);
		typename list_type::const_iterator iter = begin();
		while (iter != end())
		{
			(*iter).flatten(strm);
			iter++;
		}
	}
	
	void unflatten(std::istream &strm)
	{
		list_type::clear();
		flattenable_uint32 nitems;
		nitems.unflatten(strm);
		T tmp;
		for (unsigned int i = 0; i < nitems; i++)
		{
			tmp.unflatten(strm);
			list_type::push_back(tmp);
		}
	}

};

template <typename T> class flattenable_vector : public flattenable, public std::vector<T>
{
public:
	typedef typename std::vector<T> vector_type;
	
	flattenable_vector() {}
	flattenable_vector(const vector_type &l) : vector_type(l) {}
	virtual ~flattenable_vector() {}
	
	const flat_type_code type_code() const { return FLAT_VECTOR_TYPE; }
	const bool is_fixed_size() const { return false; }
	
	const unsigned int flattened_size() const
	{
		unsigned int flat_size = sizeof(uint32);
		if (!vector_type::empty())
		{
			typename vector_type::const_iterator iter = begin();
			if ((*iter).is_fixed_size())
				flat_size += (vector_type::size() * (*iter).flattened_size());
			else
			{
				while (iter != end())
				{
					flat_size += (*iter).flattened_size();
					iter++;
				}
			}
		}
		return flat_size;
	}
	
	void flatten(std::ostream &strm) const
	{
		flattenable_uint32 nitems = vector_type::size();
		nitems.flatten(strm);
		typename vector_type::const_iterator iter = begin();
		while (iter != end())
		{
			(*iter).flatten(strm);
			iter++;
		}
	}
	
	void unflatten(std::istream &strm)
	{
		vector_type::clear();
		flattenable_uint32 nitems;
		nitems.unflatten(strm);
		T tmp;
		for (unsigned int i = 0; i < nitems; i++)
		{
			tmp.unflatten(strm);
			vector_type::push_back(tmp);
		}
	}
};


template <typename K, typename T>
class flattenable_map : public flattenable, public std::map<K, T>
{
public:
	typedef typename std::map<K, T> map_type;
	
	flattenable_map() {}
	flattenable_map(const map_type &l) : map_type(l) {}
	virtual ~flattenable_map() {}

	const flat_type_code type_code() const { return FLAT_MAP_TYPE; }
	const bool is_fixed_size() const { return false; }
	
	const unsigned int flattened_size() const
	{
		unsigned int flat_size = sizeof(uint32);
		K kmp;
		T tmp;
		if (kmp.is_fixed_size() && tmp.is_fixed_size())
			flat_size += ((map_type::size() * kmp.flattened_size()) +
				(map_type::size() * tmp.flattened_size()));
		else
		{
			typename map_type::const_iterator iter = begin();
			while (iter != end())
			{
				flat_size += iter->first.flattened_size();
				flat_size += iter->second.flattened_size();
				iter++;
			}
		}
		return flat_size;
	}
	
	void flatten(std::ostream &strm) const
	{
		flattenable_uint32 nitems = map_type::size();
		nitems.flatten(strm);
		typename map_type::const_iterator iter = begin();
		while (iter != end())
		{
			iter->first.flatten(strm);
			iter->second.flatten(strm);
			iter++;
		}
	}
	
	void unflatten(std::istream &strm)
	{
		map_type::clear();
		flattenable_uint32 nitems;
		nitems.unflatten(strm);
		K kmp;
		T tmp;
		for (unsigned int i = 0; i < nitems; i++)
		{
			kmp.unflatten(strm);
			tmp.unflatten(strm);
			map_type::insert(make_pair(kmp, tmp));
		}
	}
};

} // end namespace flat

#endif // FLATTYPES_H