/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.container;

import java.util.Map;
import java.util.concurrent.ConcurrentMap;


/**
 * オブジェクトコンテナのひとつのコンポーネント定義を表す。
 * 本クラスのインスタンスは複数のスレッドで同時に使用できる。
 * @author nakamura
 *
 */
abstract public class ComponentDef { 
	private final ComponentScope scope;
	private final Object singletonValue;
	
	/**
	 * コンストラクタ。
	 * @param scope スコープ。
	 * @throws NullPointerException 引数が null の場合。
	 * @throws RuntimeException スコープが{@link ComponentScope#SINGLETON}かつ{@link ComponentDef#newInstance()}が投げた場合。
	 */
	public ComponentDef(final ComponentScope scope){
		scope.getClass();
		this.scope = scope;
		if(scope == ComponentScope.SINGLETON){
			singletonValue = newInstance();
		}else{
			singletonValue = null;
		}
	}
	
	Object getComponent(final String key, final ScopeContainer scopeContainer){
		if(scope == ComponentScope.SINGLETON){
			return singletonValue;
		}
		if(scope == ComponentScope.PROTOTYPE){
			return newInstance();
		}
		if(scope == ComponentScope.REQUEST){
			return createPerScope(key, scopeContainer.getRequest());
		}else if(scope == ComponentScope.SESSION){
			return createPerScope(key, scopeContainer.getSession());
		}else{
			// APPLICATION
			return createPerScope(key, scopeContainer.getApplication());
		}
	}

	private Object createPerScope(final String key, final Map<String,Object> scope){
		while(true){
			final Object result = scope.get(key);
			if(result != null){
				return result;
			}
			scope.put(key, newInstance());
		}
	}

	private Object createPerScope(final String key, final ConcurrentMap<String,Object> scope){
		while(true){
			final Object result = scope.get(key);
			if(result != null){
				return result;
			}
			scope.putIfAbsent(key, newInstance());
		}
	}
	
	/**
	 * オブジェクトを生成するために呼び出される(called)。
	 * 実装クラスでオブジェクトの生成とパラメータの注入を行う。
	 * @return 生成したオブジェクト。
	 */
	abstract protected Object newInstance();
}
