/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.LinkedHashMap;
import java.util.Map;

import org.w3c.dom.Node;

/**
 * DOM ノードをキャッシュする{@link DomExpression}。
 * 委譲先で生成される{@link Node}が{@link #getCacheKey(String)}に対して一意であることを想定しているため並行性制御を行っていない。
 * @author nakamura
 *
 */
public class CacheNode implements DomExpression {
	private final DomExpression nodeMaker;
	private final Map<Object,Node> cacheMap;

	/**
	 * コンストラクタ。
	 * @param cacheMap キャッシュ。本クラスはこの引数の状態を変化させる。null が指定された場合はキャッシュを行わない。{@link LinkedHashMap#removeEldestEntry(java.util.Map.Entry)}を指定した{@link LinkedHashMap}を指定するとキャッシュの容量を制御することができる。{@link java.util.HashMap}などの強参照の{@link Map}を指定すると容量制限のないキャッシュになる。{@link java.util.WeakHashMap}を指定すると弱参照によるキャッシュになる。
	 * @param nodeMaker DOM ノードを生成し加工する委譲先。
	 * @throws NullPointerException nodeMaker が null の場合。
	 */
	public CacheNode(final Map<Object,Node> cacheMap, final DomExpression nodeMaker){
		nodeMaker.getClass();
		this.cacheMap = cacheMap;
		this.nodeMaker = nodeMaker;
	}

	/**
	 * @throws NullPointerException 引数または生成された{@link DomContext#getNode()}が null の場合。
	 */
	public void interpret(final DomContext context) {
		if(cacheMap == null){
			nodeMaker.interpret(context);
		}else{
			final Object key = getCacheKey(context.getId());
			Node node = cacheMap.get(key);
			if(node == null){
				nodeMaker.interpret(context);
				node = context.getNode();
				cacheMap.put(key, node);
			}
			context.setNode(node.cloneNode(true));
		}
	}
	
	/**
	 * 委譲先で生成する DOM ノードを一意に識別するキーを取得するために呼び出される(called)。
	 * デフォルトは{@link java.util.WeakHashMap}により GC されるように id でなく new String(id) を返す。
	 * {@link java.util.Locale}や端末ごとに DOM ノードをキャッシュする際にオーバライドする。
	 * オーバライドする際は DOM ノード生成クラスと整合性を保つ必要がある。
	 * @param id 委譲先で DOM ノードを生成するのに使用した id。
	 * @return 委譲先で生成する DOM ノードを一意に識別するキー。
	 */
	public Object getCacheKey(final String id){
		// http://www.java-tips.org/java-se-tips/java.util/using-weakhashmap-for-listener-lists.html
		return new String(id);
	}
}
