/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;


import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.utils.UtilsConstants;

/**
 * {@link DomContext}からコンポーネントの一覧を取得し、
 * コンポーネントひとつひとつに対し DOM ノードをコピーして処理を委譲し、
 * 最後にオリジナルのノードを削除する{@link DomExpression}。
 * @author nakamura
 *
 */
public class Loop implements DomExpression {
	private final GettingEL collectionEL;
	private final EL valueEL;
	private final DomExpression expression;

	/**
	 * コンストラクタ。
	 * @param collectionEL コンポーネントの一覧への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param valueEL コンポーネントひとつひとつの格納先への参照。
	 * @param expression コンポーネントひとつひとつに対し DOM ノードをコピーした後の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public Loop(final GettingEL collectionEL, final EL valueEL, final DomExpression expression){
		collectionEL.getClass();
		valueEL.getClass();
		expression.getClass();
		
		this.collectionEL = collectionEL;
		this.valueEL = valueEL;
		this.expression = expression;
	}

	/**
	 * @throws NullPointerException 引数または{@link DomContext#getNode()}が null の場合。
	 * @throws ClassCastException {@link DomContext#getNode()}が{@link Element}でない場合。
	 */
	public void interpret(final DomContext context) {
		final Object value1 = collectionEL.getValue(context);
		final Iterable iterable = UtilsConstants.toIterable(value1);
		for(final Object value:iterable){
			final Node templateNode = context.getNode();
			final Node parentNode = templateNode.getParentNode();

			final Node newNode = templateNode.cloneNode(true);
			final DomContext newContext = context.copy();
			newContext.setNode(newNode);
			valueEL.setValue(newContext, value);
			expression.interpret(newContext);
			parentNode.insertBefore(newNode, templateNode);
		}
		DomConstants.removeThis(context.getNode());
	}
}
