/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Semaphore;


/**
 * セッション毎にセマフォ(同時実行スレッド数)を制御する{@link DomExpression}。
 * 同時実行数を越えている場合のリクエストはスレッドをブロックせずにエラーとする。
 * HTML フォームを使用する際はクライアントからの並行なリクエストをエラーにする可能性があるため使用に注意する必要がある。
 * @author nakamura
 *
 */
public class TrySemaphoreSession implements DomExpression {
	/**
	 * セマフォを格納する{@link DomContext#getSession()}のキーのデフォルト値。
	 */
	public static final String KEY = "woolpack.dom.TRY_SEMAPHORE";
	
	private final String key;
	private final int permits;
	private final DomExpression trueExpression;
	private final DomExpression falseExpression;
	
	/**
	 * コンストラクタ。
	 * @param key セマフォを格納する{@link DomContext#getSession()}のキー。
	 * @param permits 用可能なパーミットの初期数。
	 * @param trueExpression パーミットの取得に成功した場合の委譲先。
	 * @param falseExpression パーミットの取得に失敗した場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public TrySemaphoreSession(
			final String key, 
			final int permits,
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		key.charAt(0);
		trueExpression.getClass();
		falseExpression.getClass();

		this.key = key;
		this.permits = permits;
		this.trueExpression = trueExpression;
		this.falseExpression = falseExpression;
	}

	/**
	 * コンストラクタ。
	 * セマフォを格納する{@link DomContext#getSession()}のキーとして{@link #KEY}を使用する。
	 * @param permits 用可能なパーミットの初期数。
	 * @param trueExpression パーミットの取得に成功した場合の委譲先。
	 * @param falseExpression パーミットの取得に失敗した場合の委譲先。
	 * @throws NullPointerException 引数のいずれかが null の場合。
	 */
	public TrySemaphoreSession(
			final int permits,
			final DomExpression trueExpression, 
			final DomExpression falseExpression){
		this(KEY, permits, trueExpression, falseExpression);
	}
	
	private Semaphore createPerSession(final ConcurrentMap<String,Object> session){
		while(true){
			final Semaphore semaphore = (Semaphore)session.get(key);
			if(semaphore != null){
				return semaphore;
			}
			session.putIfAbsent(key, new Semaphore(permits, true));
		}
	}

	public void interpret(final DomContext context) {
		final Semaphore semaphore = createPerSession(context.getSession());
		if(semaphore.tryAcquire()){
			try{
				trueExpression.interpret(context);
			}finally{
				semaphore.release();
			}
		}else{
			falseExpression.interpret(context);
		}
	}
}
