/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.lang.reflect.Array;
import java.util.Collection;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;

import woolpack.utils.UtilsConstants;

/**
 * 一覧型をコンバートする{@link GettingEL}。
 * @author nakamura
 *
 */
public class CollectionTypeConvertEL extends GettingELAdapter {
	private final GettingEL convertEL;
	
	/**
	 * コンストラクタ。
	 * @param convertEL 委譲先。単純型のコンバートするために使用する。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public CollectionTypeConvertEL(final GettingEL convertEL){
		convertEL.getClass();
		this.convertEL = convertEL;
	}

	@Override
	public Object getValue(Object beforeValue, Class toType)
			throws ELTargetRuntimeException {
		if(Object.class.equals(toType)){
			return beforeValue;
		}
		final Collection collection = UtilsConstants.toCollection(beforeValue);
		if(toType.isArray()){
			// TODO 変換元が配列であっても新しい配列が作成されるので、再検討する。
			final Object afterArray = Array.newInstance(toType.getComponentType(), collection.size());
			int i=0;
			for(final Object before:collection){
				// TODO 値が null の場合にIllegalArgumentException が投げられる。
				final Object after = convertEL.getValue(before, toType.getComponentType());
				Array.set(afterArray, i, after);
				i++;
			}
			return afterArray;
		}else if(Collection.class.isAssignableFrom(toType)){
			return collection;
		}else{
			final Object value = collection.iterator().next();
			if(value != null){
				return convertEL.getValue(value, toType);
			}else{
				return value;
			}
		}
	}

	@Override public int hashCode(){
		return new HashCodeBuilder().append(convertEL).toHashCode();
	}
	
	@Override public boolean equals(final Object obj){
		if(!(obj instanceof CollectionTypeConvertEL)){
			return false;
		}
		if(this == obj){
			return true;
		}
		final CollectionTypeConvertEL o = (CollectionTypeConvertEL)obj;
		return new EqualsBuilder()
		.append(this.convertEL, o.convertEL)
		.isEquals();
	}
	
	@Override public String toString(){
		return new ToStringBuilder(this).append("convertEL", convertEL).toString();
	}
}
