/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Arrays;
import java.util.Map;

import woolpack.dom.Branch;
import woolpack.dom.BranchByAttrValue;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.EvalAttrValueContains;
import woolpack.dom.If;
import woolpack.dom.RemoveAttr;
import woolpack.dom.UpdateAttrValue;
import woolpack.dom.XPath;
import woolpack.el.FixEL;
import woolpack.el.GettingEL;
import woolpack.utils.SwitchBuilder;


/**
 * 値を再生成する{@link DomExpression}。
 * 入力部品(file, image)の場合はなにもしない。
 * ノードの種類がラジオボタン/チェックボックス/セレクトで selectFlag の場合、selected 属性の有無または checked 属性の有無を変更する。
 * ノードの種類がラジオボタン/チェックボックス/セレクトで selectFlag でないか、
 * 入力部品(text, password, hidden, submit, reset, button)の場合、value 属性値を変更する。
 * ノードの種類が入力部品以外であるかテキストエリアの場合、子ノードを値のテキストで置き換える。
 * @author nakamura
 *
 */
public class UpdateValue implements DomExpression {
	private static final String VALUE = "value";
	private static final String SELECTED = "selected";
	private static final String CHECKED = "checked";
	private static final GettingEL TRUE_EL = new FixEL("true");

	private final DomExpression root;
	
	/**
	 * コンストラクタ。
	 * @param valueEL 値の取得先への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param mapEL 値とラベルの{@link Map}への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param selectFlag selected または checked 属性の有無を変更するなら true。value の属性値を変更するなら false。
	 * @throws NullPointerException valueEL が null の場合。
	 */
	public UpdateValue(
			final GettingEL valueEL, 
			final GettingEL mapEL, 
			final boolean selectFlag){
		
		final DomExpression updateValueAttr = new UpdateAttrValue(VALUE, valueEL);
		
		final DomExpression expressionForOption = selectFlag?
				new If(
						new EvalAttrValueContains(VALUE, valueEL), 
						new UpdateAttrValue(SELECTED, TRUE_EL),
						new RemoveAttr(SELECTED)
						)
				:updateValueAttr;
		root = new Branch<String>(DomConstants.EVAL_NODE_NAME,
				new SwitchBuilder<String,DomExpression>()
				.put("OPTION", expressionForOption)
				.put("SELECT", new XPath(HtmlConstants.XPATH_OPTION, expressionForOption))
				.put("INPUT", new BranchByAttrValue(Arrays.asList("type"), 
						new SwitchBuilder<String,DomExpression>()
						.put("file", DomConstants.NULL)
						.put("image")
						.get(selectFlag?
								new BranchByAttrValue(Arrays.asList("type"), 
										new SwitchBuilder<String,DomExpression>()
										.put("radio", new If(
												new EvalAttrValueContains(VALUE, valueEL),  
												new UpdateAttrValue(CHECKED, TRUE_EL),
												new RemoveAttr(CHECKED)))
										.put("checkbox")
										.get(updateValueAttr))
								:updateValueAttr)
				))
				.put("BUTTON", updateValueAttr)
				.get(new UpdateToSelectedValue(valueEL, mapEL)));
	}
	
	/**
	 * コンストラクタ。
	 * @param valueEL 値の取得先。
	 * @param mapEL 値とラベルの{@link Map}への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @throws NullPointerException valueEL が null の場合。
	 */
	public UpdateValue(final GettingEL valueEL, final GettingEL mapEL){
		this(valueEL, mapEL, true);
	}
	
	/**
	 * コンストラクタ。
	 * @param valueEL 値の取得先への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public UpdateValue(final GettingEL valueEL){
		this(valueEL, null, true);
	}
	
	public void interpret(final DomContext context) {
		root.interpret(context);
	}
}
