/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.text.DecimalFormat;
import java.text.FieldPosition;
import java.text.NumberFormat;
import java.text.ParsePosition;

/**
 * フォーマット処理を委譲する変換器。
 * @author nakamura
 *
 */
public class DelegateNumberFormat extends NumberFormat {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private final NumberFormat format;
	private final String toStringString;
	
	/**
	 * コピーコンストラクタ。
	 * {@link #clone()}から呼び出される(called)。
	 * @param format コピー元。
	 * @param dummy ダミー。
	 */
	protected DelegateNumberFormat(final DelegateNumberFormat format, final boolean dummy){
		this.format = format.format;
		this.toStringString = format.toStringString;
	}

	/**
	 * コンストラクタ。
	 * @param format 委譲先。
	 * @param toStringString {@link #toString()}の返却値。
	 * @throws NullPointerException format が null の場合。
	 */
	public DelegateNumberFormat(final NumberFormat format, final String toStringString){
		format.getClass();
		this.format = format;
		this.toStringString = toStringString;
	}
	
	/**
	 * コンストラクタ。
	 * @param format 委譲先。
	 * @throws NullPointerException 引数が null の場合。
	 */
	public DelegateNumberFormat(final NumberFormat format){
		this(format, null);
	}

	/**
	 * コンストラクタ。
	 * {@link DecimalFormat}を委譲先とする。
	 *
	 */
	public DelegateNumberFormat(){
		this(new DecimalFormat());
	}

	@Override
	public StringBuffer format(final double number, final StringBuffer toAppendTo,
			final FieldPosition pos) {
		return format.format(number, toAppendTo, pos);
	}

	@Override
	public StringBuffer format(final long number, final StringBuffer toAppendTo,
			final FieldPosition pos) {
		return format.format(number, toAppendTo, pos);
	}

	@Override
	public Number parse(final String source, final ParsePosition parsePosition) {
		return format.parse(source, parsePosition);
	}

	@Override public Object clone(){
		return new DelegateNumberFormat(this, false);
	}
	
	@Override public String toString(){
		if(toStringString != null){
			return toStringString;
		}
		return format.toString();
	}
}
