/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

/**
 * {@link ForwardMatchable}のデフォルト実装。 コンストラクタ引数により以下の判定を行う。
 * null の場合は、メソッドが返却したオブジェクトが null であることを検証する。
 * {@link Class}クラスのインスタンスの場合は、メソッドが返却したオブジェクトが
 * コンストラクタ引数が表すオブジェクトと代入互換であることを検証する。
 * 上記以外の場合は、値が等しいことを検証する。
 * 
 * @author nakamura
 * 
 */
public class ForwardMatcher implements ForwardMatchable {
	private final Object returnedObject;

	private final ForwardMatchable base;

	/**
	 * コンストラクタ。
	 * 
	 * @param returnedObject
	 *            メソッドが返却したオブジェクトを判定するためのオブジェクト。
	 */
	public ForwardMatcher(final Object returnedObject) {
		this.returnedObject = returnedObject;

		if (returnedObject == null) {
			base = new ForwardMatchable() {
				public boolean matches(final Object aReturnedObject) {
					return aReturnedObject == null;
				}
			};
		} else if (returnedObject instanceof Class) {
			base = new ForwardMatchable() {
				public boolean matches(final Object aReturnedObject) {
					return ((Class) returnedObject).isInstance(aReturnedObject);
				}
			};
		} else {
			base = new ForwardMatchable() {
				public boolean matches(final Object aReturnedObject) {
					return returnedObject.equals(aReturnedObject);
				}
			};
		}
	}

	public boolean matches(final Object aReturnedObject) {
		return base.matches(aReturnedObject);
	}

	/**
	 * メソッドが返却したオブジェクトを判定するオブジェクトを返す。
	 * 
	 * @return メソッドが返却したオブジェクトを判定するオブジェクト。
	 */
	public Object getReturnedObject() {
		return returnedObject;
	}
}
