/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.adapter;

import java.util.List;

import org.apache.commons.jxpath.JXPathContext;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.utils.CheckUtils;
import woolpack.utils.NodeFindable;

/**
 * JXPath を使用して DOM ノードを検索する{@link NodeFindable}。 本クラスはイミュータブルである。 本クラスは
 * JXPath のライブラリを使用する。 委譲先でノードを操作した場合、次の検索結果に影響する(スナップショットではない)。
 * 適用しているパターン：Adapter。
 * 
 * @author nakamura
 * 
 */
public class JXP implements NodeFindable {
	private final String criteria;

	/**
	 * コンストラクタ。
	 * 
	 * @param criteria
	 *            検索条件。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             引数が空の場合。
	 * @throws org.apache.commons.jxpath.JXPathException
	 *             のコンパイルに失敗した場合。
	 */
	public JXP(final String criteria) {
		CheckUtils.checkNotEmpty(criteria);
		this.criteria = criteria;
		JXPathContext.compile(criteria);
	}

	/**
	 * コンストラクタ。 new JXP("//*[@" + attrName + "=\"" + attrValue + "\"]") と同一。
	 * 
	 * @param attrName
	 *            属性名。
	 * @param attrValue
	 *            属性値。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws org.apache.commons.jxpath.JXPathException
	 *             XPath のコンパイルに失敗した場合。
	 */
	public JXP(final String attrName, final String attrValue) {
		this("//*[@" + attrName + "=\"" + attrValue + "\"]");
		CheckUtils.checkNotNull(attrName);
		CheckUtils.checkNotNull(attrValue);
	}

	public NodeList evaluateList(final Object node) {
		final List list = JXPathContext.newContext(node).selectNodes(criteria);
		return new NodeList() {
			public Node item(final int index) {
				return (Node) list.get(index);
			}

			public int getLength() {
				return list.size();
			}
		};
	}

	public Node evaluateOne(final Object node) {
		return (Node) JXPathContext.newContext(node).selectSingleNode(criteria);
	}

	public String getCriteria() {
		return criteria;
	}
}
