/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.config;

import java.util.LinkedList;

import woolpack.utils.CheckUtils;

/**
 * {@link #interpret(ConfigContext)}が呼び出された回数を数えるテスト用の{@link ConfigExpression}。
 * このクラスはイミュータブルでもスレッドセーフでもない。
 * {@link #interpret(ConfigContext)}では{@link ConfigContext}に対して状態を変更する操作を行わない。
 * 適用しているパターン：Proxy。
 * 
 * @author nakamura
 * 
 */
public class ConfigCount implements ConfigExpression {
	private static final int MAX = 4;

	private final int max;

	private final ConfigExpression expression;

	private int count;

	private final LinkedList<ConfigContext> contextList;

	/**
	 * コンストラクタ。
	 * 
	 * @param max
	 *            一覧に保持する{@link ConfigContext}の最大件数。
	 * @param expression
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public ConfigCount(final int max, final ConfigExpression expression) {
		CheckUtils.checkNotNull(expression);
		this.max = max;
		this.expression = expression;
		contextList = new LinkedList<ConfigContext>();
		count = 0;
	}

	/**
	 * コンストラクタ。 一覧に保持する{@link ConfigContext}の最大件数を 4 とする。
	 * 
	 * @param expression
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public ConfigCount(final ConfigExpression expression) {
		this(MAX, expression);
	}

	public void interpret(final ConfigContext context) {
		count++;
		if (max > 0) {
			if (count > max) {
				contextList.removeFirst();
			}
			contextList.add(context);
		}
		expression.interpret(context);
	}

	/**
	 * {@link #interpret(ConfigContext)}が呼び出された回数を返す。
	 * 
	 * @return {@link #interpret(ConfigContext)}が呼び出された回数。
	 */
	public int getCount() {
		return count;
	}

	/**
	 * 呼び出された{@link #interpret(ConfigContext)}の引数{@link ConfigContext}の一覧を返す。
	 * 
	 * @return 呼び出された{@link #interpret(ConfigContext)}
	 * の引数{@link ConfigContext}の一覧。
	 */
	public LinkedList<ConfigContext> getContextList() {
		return contextList;
	}
}
