/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.container;

import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import woolpack.utils.CheckUtils;

/**
 * オブジェクトコンテナのひとつのコンポーネント定義を表す。 適用しているパターン：Template Method。
 * 
 * @author nakamura
 * 
 */
public abstract class AbstractComponentDef {
	private final ComponentScope scope;

	private final Object singletonValue;

	/**
	 * コンストラクタ。
	 * 
	 * @param scope
	 *            スコープ。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 * @throws RuntimeException
	 *             スコープが{@link ComponentScope#SINGLETON}かつ
	 *             {@link AbstractComponentDef#newInstance()}が例外を投げた場合。
	 */
	public AbstractComponentDef(final ComponentScope scope) {
		CheckUtils.checkNotNull(scope);
		this.scope = scope;
		if (scope == ComponentScope.SINGLETON) {
			singletonValue = newInstance();
		} else {
			singletonValue = null;
		}
	}

	Object getComponent(final String key, final ScopeContainer scopeContainer) {
		if (scope == ComponentScope.SINGLETON) {
			return singletonValue;
		}
		if (scope == ComponentScope.PROTOTYPE) {
			return newInstance();
		}
		if (scope == ComponentScope.REQUEST) {
			return createPerScope(key, scopeContainer.getRequest());
		} else if (scope == ComponentScope.SESSION) {
			return createPerScope(key, scopeContainer.getSession());
		} else {
			// APPLICATION
			return createPerScope(key, scopeContainer.getApplication());
		}
	}

	private Object createPerScope(final String key,
			final Map<String, Object> map) {
		while (true) {
			final Object result = map.get(key);
			if (result != null) {
				return result;
			}
			map.put(key, newInstance());
		}
	}

	private Object createPerScope(final String key,
			final ConcurrentMap<String, Object> map) {
		while (true) {
			final Object result = map.get(key);
			if (result != null) {
				return result;
			}
			map.putIfAbsent(key, newInstance());
		}
	}

	/**
	 * オブジェクトを生成するために呼び出される(called)。 実装クラスでオブジェクトの生成とパラメータの注入を行う。
	 * 本クラスのインスタンスは複数のスレッドで同時に使用できる必要がある。
	 * 
	 * @return 生成したオブジェクト。
	 */
	protected abstract Object newInstance();
}
