/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.LinkedList;

import woolpack.utils.CheckUtils;

/**
 * {@link #interpret(DomContext)}が呼び出された回数を数えるテスト用{@link DomExpression}。
 * このクラスはイミュータブルでもスレッドセーフでもない。
 * {@link #interpret(DomContext)}では{@link DomContext}に対して状態を変更する操作を行わない。
 * 適用しているパターン：Proxy。
 * 
 * @author nakamura
 * 
 */
public class Count implements DomExpression {
	private static final int MAX = 4;

	private final int max;

	private final DomExpression expression;

	private int count;

	private final LinkedList<DomContext> contextList;

	/**
	 * コンストラクタ。
	 * 
	 * @param max
	 *            一覧に保持する{@link DomContext}の最大件数。
	 * @param expression
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public Count(final int max, final DomExpression expression) {
		CheckUtils.checkNotNull(expression);
		this.max = max;
		this.expression = expression;
		contextList = new LinkedList<DomContext>();
		count = 0;
	}

	/**
	 * コンストラクタ。 一覧に保持する{@link DomContext}の最大件数 4 とする。
	 * 
	 * @param expression
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public Count(final DomExpression expression) {
		this(MAX, expression);
	}

	public void interpret(final DomContext context) {
		count++;
		if (max > 0) {
			if (count > max) {
				contextList.removeFirst();
			}
			contextList.add(context);
		}
		expression.interpret(context);
	}

	/**
	 * {@link #interpret(DomContext)}が呼び出された回数を返す。
	 * 
	 * @return {@link #interpret(DomContext)}が呼び出された回数。
	 */
	public int getCount() {
		return count;
	}

	/**
	 * 呼び出された{@link #interpret(DomContext)}の引数の一覧を返す。
	 * 
	 * @return 呼び出された{@link #interpret(DomContext)}の引数の一覧。
	 */
	public LinkedList<DomContext> getContextList() {
		return contextList;
	}
}
