/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.io.IOException;

import woolpack.utils.CheckUtils;

/**
 * 機能を逸脱するエラーが発生した場合にコンテキストのダンプ情報を付加した
 * {@link IllegalStateException}でラップして委譲元に例外を投げる{@link DomExpression}。
 * 適用しているパターン：Proxy。
 * 
 * @author nakamura
 * 
 */
public class DumpIfCatch implements DomExpression {
	private final DomExpression expression;

	/**
	 * コンストラクタ。
	 * 
	 * @param expression
	 *            委譲先。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public DumpIfCatch(final DomExpression expression) {
		CheckUtils.checkNotNull(expression);
		this.expression = expression;
	}

	/**
	 * @throws IllegalStateException
	 *             委譲先で{@link RuntimeException}が発生した場合。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public void interpret(final DomContext context) {
		try {
			expression.interpret(context);
		} catch (final RuntimeException e) {
			final StringBuilder sb = new StringBuilder();
			try {
				context.appendTo(sb);
			} catch (final IOException e1) {
				sb.append(e1); // カバレージがここを通過してはいけない
			}
			throw new IllegalStateException(sb.toString(), e);
		}
	}
}
