/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.text.Format;

import org.w3c.dom.Element;

import woolpack.text.CloneFormatFactory;
import woolpack.text.FormatFactory;
import woolpack.utils.CheckUtils;

/**
 * {@link Format#format(java.lang.Object)}を使用して属性値を変換する{@link DomExpression}。
 * 
 * @author nakamura
 * 
 */
public class FormatAttrValue implements DomExpression {
	private final String attrName;

	private final FormatFactory factory;

	/**
	 * コンストラクタ。
	 * 
	 * @param attrName
	 *            属性名。
	 * @param factory
	 *            フォーマットのファクトリ。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             attrName が空の場合。
	 */
	public FormatAttrValue(final String attrName, final FormatFactory factory) {
		CheckUtils.checkNotEmpty(attrName);
		CheckUtils.checkNotNull(factory);

		this.attrName = attrName;
		this.factory = factory;
	}

	/**
	 * コンストラクタ。 format を{@link Format#clone()}で複製して利用する。
	 * 
	 * @param attrName
	 *            属性名。
	 * @param format
	 *            フォーマット。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             attrName が空の場合。
	 */
	public FormatAttrValue(final String attrName, final Format format) {
		this(attrName, new CloneFormatFactory(format));
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link DomContext#getNode()}が null の場合。
	 * @throws ClassCastException
	 *             {@link DomContext#getNode()}が{@link Element}でない場合。
	 */
	public void interpret(final DomContext context) {
		final Element e = (Element) context.getNode();
		final String before = e.getAttribute(attrName);
		final String after = factory.newInstance().format(before);
		e.setAttribute(attrName, after);
	}
}
