/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.text.Format;

import woolpack.text.CloneFormatFactory;
import woolpack.text.FormatFactory;
import woolpack.utils.CheckUtils;

/**
 * {@link Format#format(java.lang.Object)}を使用して
 * {@link DomContext#getId()}を変換する{@link DomExpression}。
 * 
 * @author nakamura
 * 
 */
public class FormatId implements DomExpression {
	private final FormatFactory factory;

	/**
	 * コンストラクタ。
	 * 
	 * @param factory
	 *            フォーマットのファクトリ。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public FormatId(final FormatFactory factory) {
		CheckUtils.checkNotNull(factory);
		this.factory = factory;
	}

	/**
	 * コンストラクタ。 format を{@link Format#clone()}で複製して利用する。
	 * 
	 * @param format
	 *            フォーマット。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public FormatId(final Format format) {
		this(new CloneFormatFactory(format));
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link DomContext#getId()}が null の場合。
	 */
	public void interpret(final DomContext context) {
		context.setId(factory.newInstance().format(context.getId()));
	}
}
