/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import woolpack.utils.CheckUtils;

/**
 * {@link DomContext#getInput()}をセッションに保持し再利用する{@link DomExpression}。
 * 適用しているパターン：Proxy。
 * 
 * @author nakamura
 * 
 */
public class SaveInputForBack implements DomExpression {
	/**
	 * id と入力{@link Map}の{@link Map}を保持する
	 * {@link DomContext#getSession()}上の位置のデフォルト値。
	 */
	public static final String KEY = "woolpack.dom.SAVE_INPUT_FOR_BACK";

	private final String key;

	private final DomExpression expression;

	private final Map<String, String> toRequestIdMap;

	private final Map<String, String> toInputIdMap;

	/**
	 * コンストラクタ。
	 * 
	 * @param key
	 *            id と入力{@link Map}の{@link Map}を保持する
	 *            {@link DomContext#getSession()}上の位置。
	 * @param expression
	 *            委譲先。
	 * @param toRequestIdMap
	 *            リクエストされた id と変換される id の{@link Map}。
	 * @param toInputIdMap
	 *            リクエストされた id と再利用する
	 *            {@link DomContext#getInput()}に対応する id の{@link Map}。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             key が空の場合。
	 */
	public SaveInputForBack(final String key, final DomExpression expression,
			final Map<String, String> toRequestIdMap,
			final Map<String, String> toInputIdMap) {
		CheckUtils.checkNotEmpty(key);
		CheckUtils.checkNotNull(expression);
		CheckUtils.checkNotNull(toRequestIdMap);
		CheckUtils.checkNotNull(toInputIdMap);

		this.key = key;
		this.expression = expression;
		this.toRequestIdMap = toRequestIdMap;
		this.toInputIdMap = toInputIdMap;
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param expression
	 *            委譲先。
	 * @param toRequestIdMap
	 *            リクエストされた id と変換される id の{@link Map}。
	 * @param toInputIdMap
	 *            リクエストされた id と再利用する
	 *            {@link DomContext#getInput()}に対応する id の{@link Map}。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             key が空の場合。
	 */
	public SaveInputForBack(final DomExpression expression,
			final Map<String, String> toRequestIdMap,
			final Map<String, String> toInputIdMap) {
		this(KEY, expression, toRequestIdMap, toInputIdMap);
	}

	private Map<String, Map> createPerSession(
			final ConcurrentMap<String, Object> session) {
		while (true) {
			Map<String, Map> map = (Map<String, Map>) session.get(key);
			if (map != null) {
				return map;
			}
			map = Collections.synchronizedMap(new HashMap<String, Map>());
			session.putIfAbsent(key, map);
		}
	}

	public void interpret(final DomContext context) {
		final Map<String, Map> map = createPerSession(context.getSession());
		{
			final String inputId = toInputIdMap.get(context.getId());
			if (inputId != null) {
				final Map m = map.get(toInputIdMap.get(context.getId()));
				if (m != null) {
					context.setInput(m);
				}
			}
		}
		if (toInputIdMap.containsValue(context.getId())) {
			map.put(context.getId(), context.getInput());
		}
		{
			final String requestId = toRequestIdMap.get(context.getId());
			if (requestId != null) {
				context.setId(requestId);
			}
		}
		expression.interpret(context);
	}
}
