/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import org.w3c.dom.Element;

import woolpack.el.GettingEL;
import woolpack.utils.CheckUtils;

/**
 * 属性値を更新する{@link DomExpression}。 コンテキストから取得した値が null の場合は値を更新しない。
 * 
 * @author nakamura
 * 
 */
public class UpdateAttrValue implements DomExpression {
	private final String attrName;

	private final GettingEL valueEL;

	/**
	 * コンストラクタ。
	 * 
	 * @param attrName
	 *            属性名。
	 * @param valueEL
	 *            属性値の取得先への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public UpdateAttrValue(final String attrName, final GettingEL valueEL) {
		CheckUtils.checkNotEmpty(attrName);
		CheckUtils.checkNotNull(valueEL);

		this.attrName = attrName;
		this.valueEL = valueEL;
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link DomContext#getNode()}または取得した文字列が null の場合。
	 * @throws ClassCastException
	 *             {@link DomContext#getNode()}が{@link Element}でない場合。
	 */
	public void interpret(final DomContext context) {
		final Element element = (Element) context.getNode();
		final String value = (String) valueEL.getValue(context, String.class);
		if (value != null) {
			element.setAttribute(attrName, value);
		}
	}
}
