/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import woolpack.action.ActionInvoker;
import woolpack.action.ActionResult;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.el.ArrayPathEL;
import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.el.MapEL;
import woolpack.html.AutoUpdate;
import woolpack.utils.CheckUtils;
import woolpack.utils.NodeFindableFactory;
import woolpack.utils.UtilsConstants;

/**
 * アクションを実行する{@link DomExpression}/
 * アクションの実行結果を DOM ノードに自動生成する{@link DomExpression}/
 * アクションが返したエラーのメッセージを DOM ノードに設定する{@link DomExpression}のビルダ。
 * 適用しているパターン：Adapter, Hook Operation。
 * 
 * @author nakamura
 * 
 */
public class ActionBuilder {

	/**
	 * 遷移先で使用するコンポーネントの取得先の{@link EL}のデフォルト値。
	 */
	public static final EL DEFAULT_FORWARD_COMPONENT_EL_EL = new ArrayPathEL(
			DomConstants.LOCAL_EL, new MapEL("woolpack.ee.FORWARD_COMPONENT"));

	/**
	 * 返却値の取得先のデフォルト値。
	 */
	public static final EL DEFAULT_RETURN_EL = new ArrayPathEL(
			DomConstants.LOCAL_EL, new MapEL("returnMessages"));

	private final ActionInvoker actionDefs;

	private final Iterable<String> attrNames;

	private final EL forwardComponentELEL;

	private final EL returnEL;

	private final NodeFindableFactory factory;

	/**
	 * コンストラクタ。
	 * 
	 * @param actionDefs
	 *            アクション定義の一覧と遷移先定義の一覧の集合。
	 * @param attrNames
	 *            属性名の一覧。本クラスはこの引数の状態を変化させない。
	 * @param forwardComponentELEL
	 *            遷移先で使用するコンポーネントの設定先の{@link EL}への参照。
	 * @param returnEL
	 *            返却値の設定先への参照。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public ActionBuilder(final ActionInvoker actionDefs,
			final Iterable<String> attrNames, final EL forwardComponentELEL,
			final EL returnEL, final NodeFindableFactory factory) {
		CheckUtils.checkNotNull(actionDefs);
		CheckUtils.checkNotNull(attrNames);
		CheckUtils.checkNotNull(forwardComponentELEL);
		CheckUtils.checkNotNull(returnEL);
		CheckUtils.checkNotNull(factory);

		this.actionDefs = actionDefs;
		this.attrNames = attrNames;
		this.forwardComponentELEL = forwardComponentELEL;
		this.returnEL = returnEL;
		this.factory = factory;
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param actionDefs
	 *            アクション定義の一覧と遷移先定義の一覧の集合。
	 * @param attrNames
	 *            属性名の一覧。本クラスはこの引数の状態を変化させない。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public ActionBuilder(
			final ActionInvoker actionDefs,
			final Iterable<String> attrNames,
			final NodeFindableFactory factory) {
		this(actionDefs, attrNames, DEFAULT_FORWARD_COMPONENT_EL_EL,
				DEFAULT_RETURN_EL, factory);
	}

	/**
	 * アクションを実行し実行結果を{@link #getReturnEL()}の位置に設定する{@link DomExpression}を返す。
	 * 
	 * @return アクションを実行する{@link DomExpression}。
	 */
	public DomExpression getActionExpression() {
		return new DomExpression() {
			public void interpret(final DomContext context) {
				final ActionResult actionResult = actionDefs.invoke(context,
						context.getId(), context.getInput());
				context.setId(actionResult.getForwardDef().getId());
				forwardComponentELEL.setValue(context, actionResult
						.getForwardDef().getComponentEL());
				if (actionResult.getReturnedObject() instanceof Throwable) {
					returnEL.setValue(context, UtilsConstants
							.toMessageList((Throwable) actionResult
									.getReturnedObject()));
				} else {
					returnEL.setValue(
							context,
							actionResult.getReturnedObject());
				}
			}
		};
	}

	/**
	 * コンポーネントの取得先をアクションの実行結果から取得し、
	 * 属性値をプロパティ名としてコンポーネントから値を取得し
	 * DOM ノードに自動設定する{@link DomExpression}を返す。
	 * 
	 * @return アクションの実行結果を DOM ノードに自動生成する{@link DomExpression}。
	 */
	public DomExpression getAutoUpdateExpression() {
		return new DomExpression() {
			public void interpret(final DomContext context) {
				final GettingEL o = (GettingEL) forwardComponentELEL
						.getValue(context);
				if (o != null) {
					new AutoUpdate(attrNames, o, factory).interpret(context);
				}
			}
		};
	}

	/**
	 * 遷移先のコンポーネントの取得先への参照を返す。
	 * 
	 * @return 遷移先のコンポーネントの取得先への参照。
	 */
	public EL getForwardComponentELEL() {
		return forwardComponentELEL;
	}

	/**
	 * 返却値の取得先への参照を返す。
	 * 
	 * @return 返却値の取得先への参照。
	 */
	public EL getReturnEL() {
		return returnEL;
	}
}
