/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Map;
import java.util.Map.Entry;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.el.GettingEL;
import woolpack.utils.CheckUtils;
import woolpack.utils.NodeFindable;
import woolpack.utils.NodeFindableFactory;

/**
 * HTML のセレクトを再生成する{@link DomExpression}。 静的な選択肢は冗長性があっても静的であるほうが見える化が維持されるため、
 * 入力値を確認する画面を生成するのでなければ 静的な選択肢を生成する目的では本機能を使用しないことを推奨する。
 * 
 * @author nakamura
 * 
 */
public class MakeSelect implements DomExpression {
	private final GettingEL mapEL;

	private final NodeFindable xpathOption;

	/**
	 * コンストラクタ。
	 * 
	 * @param mapEL
	 *            値とラベルの{@link Map}への参照。
	 *            コンテキスト役に対して副作用が発生すべきではない。
	 *            キー・値とも{@link Object#toString()}で文字列として扱う。
	 * @param factory
	 *            {@link NodeFindable}のファクトリ。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public MakeSelect(
			final GettingEL mapEL,
			final NodeFindableFactory factory) {
		CheckUtils.checkNotNull(mapEL);
		this.mapEL = mapEL;
		xpathOption = factory.newInstance("//OPTION");
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link DomContext#getNode()}または取得した{@link Map}のキーか値が
	 *             null の場合。
	 * @throws ClassCastException
	 *             {@link DomContext#getNode()}が{@link Element}でない場合。
	 */
	public void interpret(final DomContext context) {
		final Map configMap;
		{
			final Object o = mapEL.getValue(context);
			if (!(o instanceof Map)) {
				return;
			}
			configMap = (Map) o;
		}

		final Node templateNode = xpathOption.evaluateOne(context.getNode());
		DomConstants.removeChildren(templateNode);

		for (final Object entryObject : configMap.entrySet()) {
			final Entry entry = (Entry) entryObject;
			final Node newNode = templateNode.cloneNode(true);
			((Element) newNode).setAttribute("value", entry.getKey()
					.toString());
			DomConstants.appendText(newNode, entry.getValue().toString());
			templateNode.getParentNode().insertBefore(newNode, templateNode);
		}
		Node nextNode = templateNode.getNextSibling();
		while (nextNode != null) {
			DomConstants.removeThis(nextNode);
			nextNode = templateNode.getNextSibling();
		}
		DomConstants.removeThis(templateNode);
	}
}
