/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Map;

import org.w3c.dom.Node;

import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.el.GettingEL;
import woolpack.utils.CheckUtils;
import woolpack.utils.UtilsConstants;

/**
 * 値に対応するラベルを再生成する{@link DomExpression}。
 * 値が複数の場合は複数のラベルを再生成する。
 * {@link Map}が存在しない場合または{@link Map}
 * に対応するキーが存在しない場合は値をそのまま表示する。
 * 前の画面のラジオボタン・チェックボックス・セレクトで選択した値を確認画面で表示するために使用する。
 * 
 * @author nakamura
 * 
 */
public class UpdateToSelectedValue implements DomExpression {
	private final GettingEL valueEL;

	private final GettingEL mapEL;

	/**
	 * コンストラクタ。 mapEL は null を許可する。
	 * {@link Map}が存在しない場合または{@link Map}
	 * に対応するキーが存在しない場合は値をそのまま表示する。
	 * 
	 * @param valueEL
	 *            値の取得先への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param mapEL
	 *            値とラベルの{@link Map}への参照。
	 *            コンテキスト役に対して副作用が発生すべきではない。
	 *            キー・値とも{@link Object#toString()}で文字列として扱う。
	 * @throws NullPointerException
	 *             valueEL が null の場合。
	 */
	public UpdateToSelectedValue(
			final GettingEL valueEL,
			final GettingEL mapEL) {
		CheckUtils.checkNotNull(valueEL);
		this.valueEL = valueEL;
		this.mapEL = mapEL;
	}

	/**
	 * @throws NullPointerException
	 *             引数または{@link DomContext#getNode()}または取得した{@link Map}のキーか値が
	 *             null の場合。
	 * 
	 */
	public void interpret(final DomContext context) {
		final Object value1 = valueEL.getValue(context);
		final Iterable iterable = UtilsConstants.toIterable(value1);
		for (final Object value : iterable) {
			if (value == null) {
				continue;
			}
			final String valueString = value.toString();

			final Node newNode = context.getNode().cloneNode(true);
			DomConstants.removeChildren(newNode);
			try {
				final Map map = (Map) mapEL.getValue(context);
				if (map.containsKey(valueString)) {
					// mapEL が null の場合もここに処理が遷移する。
					DomConstants.appendText(newNode, map.get(valueString)
							.toString());
				}
			} catch (final RuntimeException e) {
				DomConstants.appendText(newNode, valueString);
			}
			context.getNode().getParentNode().insertBefore(newNode,
					context.getNode());
		}
		DomConstants.removeThis(context.getNode());
	}
}
