/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import org.w3c.dom.Element;

import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.EvalAttrValueContains;
import woolpack.dom.If;
import woolpack.dom.RemoveAttr;
import woolpack.dom.UpdateAttrValue;
import woolpack.dom.XPath;
import woolpack.el.FixEL;
import woolpack.el.GettingEL;
import woolpack.utils.CheckUtils;
import woolpack.utils.NodeFindable;
import woolpack.utils.NodeFindableFactory;

/**
 * 値を再生成する{@link DomExpression}。 入力部品(file, image)の場合はなにもしない。
 * ノードの種類がラジオボタン/チェックボックス/セレクトで selectFlag の場合、selected 属性の有無または checked
 * 属性の有無を変更する。 ノードの種類がラジオボタン/チェックボックス/セレクトで selectFlag でないか、 入力部品(text,
 * password, hidden, submit, reset, button)の場合、value 属性値を変更する。
 * ノードの種類が入力部品以外であるかテキストエリアの場合、子ノードを値のテキストで置き換える。
 * 
 * @author nakamura
 * 
 */
public class UpdateValue2 implements DomExpression {
	private static final String VALUE = "value";

	private static final String SELECTED = "selected";

	private static final String CHECKED = "checked";

	private static final GettingEL TRUE_EL = new FixEL("true");

	private final GettingEL valueEL;

	private final GettingEL mapEL;

	private final boolean selectFlag;

	private final NodeFindable xpathOption;

	/**
	 * コンストラクタ。
	 * 
	 * @param valueEL
	 *            値の取得先への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param mapEL
	 *            値とラベルの{@link java.util.Map}への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @param selectFlag
	 *            selected または checked 属性の有無を変更するなら true。value の属性値を変更するなら
	 *            false。
	 * @throws NullPointerException
	 *             valueEL が null の場合。
	 */
	public UpdateValue2(final GettingEL valueEL, final GettingEL mapEL,
			final boolean selectFlag, final NodeFindableFactory factory) {
		CheckUtils.checkNotNull(valueEL);
		this.valueEL = valueEL;
		this.mapEL = mapEL;
		this.selectFlag = selectFlag;
		this.xpathOption = factory.newInstance("//OPTION");
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param valueEL
	 *            値の取得先。
	 * @param mapEL
	 *            値とラベルの{@link java.util.Map}への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @throws NullPointerException
	 *             valueEL が null の場合。
	 */
	public UpdateValue2(final GettingEL valueEL, final GettingEL mapEL,
			final NodeFindableFactory factory) {
		this(valueEL, mapEL, true, factory);
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param valueEL
	 *            値の取得先への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public UpdateValue2(final GettingEL valueEL,
			final NodeFindableFactory factory) {
		this(valueEL, null, true, factory);
	}

	private DomExpression getAttrValueExpression() {
		return new UpdateAttrValue(VALUE, valueEL);
	}

	private DomExpression getOptionExpression() {
		return selectFlag ? new If(new EvalAttrValueContains(VALUE, valueEL),
				new UpdateAttrValue(SELECTED, TRUE_EL),
				new RemoveAttr(SELECTED)) : getAttrValueExpression();
	}

	public void interpret(final DomContext context) {
		final Element e = (Element) context.getNode();
		final String elementName = e.getNodeName();
		if ("OPTION".equals(elementName)) {
			getOptionExpression().interpret(context);
		} else if ("SELECT".equals(elementName)) {
			new XPath(xpathOption, getOptionExpression()).interpret(context);
		} else if ("INPUT".equals(elementName)) {
			final String inputType = e.getAttribute("type");
			if ("file".equals(inputType) || "image".equals(inputType)) {
				// do nothing.
				return;
			} else if (selectFlag) {
				if ("radio".equals(inputType) || "checkbox".equals(inputType)) {
					new If(new EvalAttrValueContains(VALUE, valueEL),
							new UpdateAttrValue(CHECKED, TRUE_EL),
							new RemoveAttr(CHECKED)).interpret(context);
				} else {
					getAttrValueExpression().interpret(context);
				}
			} else {
				getAttrValueExpression().interpret(context);
			}
		} else if ("BUTTON".equals(elementName)) {
			getAttrValueExpression().interpret(context);
		} else {
			new UpdateToSelectedValue(valueEL, mapEL).interpret(context);
		}
	}
}
