/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.locale;

import java.util.Locale;
import java.util.ResourceBundle;

import woolpack.config.PutResourceBundle;
import woolpack.utils.CheckUtils;

/**
 * スレッドの{@link Locale}により{@link ResourceBundle}を読み込み
 * {@link woolpack.config.ConfigContext#setMap(java.util.Map)}
 * に追加する{@link woolpack.config.ConfigExpression}。
 * 適用しているパターン：Adapter。
 * 
 * @author nakamura
 * 
 */
public class LocalePutResourceBundle extends PutResourceBundle {
	private final ThreadLocal<Locale> threadLocal;

	/**
	 * コンストラクタ。
	 * 
	 * @param path
	 *            {@link ResourceBundle}の識別子。
	 * @param threadLocal
	 *            現在のスレッドの{@link Locale}を保持する
	 *            {@link ThreadLocal}。本クラスはこの引数の状態を変化させない。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 * @throws StringIndexOutOfBoundsException
	 *             path が空の場合。
	 */
	public LocalePutResourceBundle(final String path,
			final ThreadLocal<Locale> threadLocal) {
		super(path);
		CheckUtils.checkNotNull(threadLocal);
		this.threadLocal = threadLocal;
	}

	/**
	 * 現在のスレッドに対して threadLocal に null でない値が設定されている場合はその値を使用した
	 * {@link ResourceBundle#getBundle(java.lang.String, java.util.Locale)}
	 * の返却値を返し、
	 * そうでない場合は{@link ResourceBundle#getBundle(java.lang.String)}の返却値を返す。
	 */
	@Override
	public ResourceBundle getResourceBundle(final String path) {
		final Locale l = threadLocal.get();
		return (l != null) ? ResourceBundle.getBundle(path, l) : super
				.getResourceBundle(path);
	}
}
