/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.test;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import woolpack.dom.DomContext;
import woolpack.utils.CheckUtils;
import woolpack.utils.NodeFindableFactory;
import woolpack.utils.UtilsConstants;

/**
 * テスト用の{@link DomContext}のビルダ。
 * 
 * @author nakamura
 * 
 */
public class TestDomContextBuilder {
	/**
	 * {@link #selectEmbedded()}内で DOM ノードを検索するための属性名一覧のデフォルト値。
	 */
	public static final Iterable<String> ATTR_NAMES = Collections
			.singleton("id");

	private final Iterable<String> attrNames;

	private final Map<String, List<String>> input;

	private final DomContext context;

	private final NodeFindableFactory factory;

	/**
	 * コンストラクタ。
	 * 
	 * @param attrNames
	 *            {@link #selectEmbedded()}内で DOM ノードを検索するための属性名一覧。
	 * @param factory
	 *            {@link woolpack.utils.NodeFindable}のファクトリ。
	 * @throws NullPointerException
	 *             引数のいずれかが null の場合。
	 */
	public TestDomContextBuilder(final Iterable<String> attrNames,
			final NodeFindableFactory factory) {
		CheckUtils.checkNotNull(attrNames);
		CheckUtils.checkNotNull(factory);

		this.attrNames = attrNames;
		this.factory = factory;

		input = new HashMap<String, List<String>>();
		context = new DomContext();
		context.setConfig(new HashMap<String, Object>());
		context.setInput(input);
		context.setRequest(new HashMap<String, Object>());
		context.setSession(UtilsConstants.concurrentMap(
				new HashMap<String, Object>(), new Object()));
		context.setApplication(UtilsConstants.concurrentMap(
				new HashMap<String, Object>(), new Object()));
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param factory
	 *            {@link woolpack.utils.NodeFindable}のファクトリ。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public TestDomContextBuilder(final NodeFindableFactory factory) {
		this(ATTR_NAMES, factory);
	}

	/**
	 * {@link DomContext}を初期化する。
	 */
	public void init() {
		context.getInput().clear();
		context.getRequest().clear();
		context.setNode(null);
	}

	/**
	 * {@link DomContext}を初期化する。 引数を XPath として検索した結果が FORM エレメントまたは A
	 * エレメント(アンカー)の場合は キーと値を{@link DomContext#getInput()}に設定し、
	 * {@link DomContext#setId(String)}に action 属性値または href 属性値を設定する。
	 * 
	 * @param criteria
	 *            ノードの検索条件。
	 */
	public void initXPath(final String criteria) {
		context.getInput().clear();
		final Node n = factory.newInstance(criteria).evaluateOne(
				context.getNode());
		if (n != null && n.getNodeType() == Node.ELEMENT_NODE) {
			final Element e = (Element) n;
			if (e.getNodeName().equals("FORM")) {
				input.putAll(TestUtils.selectForm(e));
				context.setId(e.getAttribute("action"));
			} else if (e.getNodeName().equals("A")) {
				final String href = e.getAttribute("href");
				input.putAll(TestUtils.selectQuery(href));
				context.setId(href.substring(0, href.indexOf('?') >= 0 ? href
						.indexOf('?') : href.length()));
			}
		}
		context.getRequest().clear();
		context.setNode(null);
	}

	/**
	 * {@link DomContext#getInput()}を返す。
	 * 
	 * @return {@link DomContext#getInput()}。
	 */
	public Map<String, List<String>> getInput() {
		return input;
	}

	/**
	 * {@link DomContext}を返す。
	 * 
	 * @return {@link DomContext}。
	 */
	public DomContext get() {
		return context;
	}

	/**
	 * {@link DomContext#getNode()}を標準出力に出力する。
	 * 
	 */
	public void print() {
		TestUtils.print(context);
	}

	/**
	 * 引数で指定したノードが存在する場合は true、それ以外の場合は false を返す。
	 * 
	 * @param criteria
	 *            XPath。
	 * @return 引数で指定したノードが存在する場合は true。それ以外の場合は false。
	 */
	public boolean exists(final String criteria) {
		return selectNode(criteria) != null;
	}

	/**
	 * DOM ノードを検索する。
	 * 
	 * @param criteria
	 *            XPath。
	 * @return 検索されたDOM ノード。該当するノードが存在しない場合は null。
	 */
	public Node selectNode(final String criteria) {
		return factory.newInstance(criteria).evaluateOne(context.getNode());
	}

	/**
	 * 埋め込まれたキーと値の{@link Map}を返す。
	 * 
	 * @return 埋め込まれたキーと値の{@link Map}。
	 */
	public Map<String, List<String>> selectEmbedded() {
		return TestUtils.selectEmbedded(attrNames, context.getNode());
	}
}
