/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import woolpack.utils.CheckUtils;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.Switchable;

/**
 * 有限個の文字列を文字列にフォーマットする変換器。
 * {@link #format(Object, StringBuffer, FieldPosition)}
 * ・{@link #parseObject(String, ParsePosition)}
 * とも同一の変換を行うため、可逆ではない。 このクラスは入力チェックで使用することを想定していない。 このクラスは対応する Javascript
 * クラスを持たない。
 * 
 * @author nakamura
 * 
 */
public class SwitchableFormat extends Format {
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Switchable<String, String> switchable;
	
	private void writeObject(final ObjectOutputStream out) throws IOException {
		final LinkedHashMap<String, String> map = new LinkedHashMap<String, String>();
		for (final String s : switchable.keys()) {
			map.put(s, switchable.get(s));
		}
		out.writeObject(map);
		out.writeObject(switchable.getDefault());
	}
	
	private void readObject(final ObjectInputStream in)
	throws IOException, ClassNotFoundException {
		final Map map = (Map) in.readObject();
		final String defaultValue = (String) in.readObject();
		final SwitchBuilder<String, String> b
		= new SwitchBuilder<String, String>();
		for (final Object entryObject : map.entrySet()) {
			final Entry entry = (Entry) entryObject;
			b.put((String) entry.getKey(), (String) entry.getValue());
		}
		switchable = b.get(defaultValue);
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param switchable
	 *            変換元と変換先の{@link Switchable}。
	 * @throws NullPointerException
	 *             parseMap が null の場合。
	 */
	public SwitchableFormat(final Switchable<String, String> switchable) {
		super();
		CheckUtils.checkNotNull(switchable);
		this.switchable = switchable;
	}

	@Override
	public StringBuffer format(final Object obj, final StringBuffer toAppendTo,
			final FieldPosition pos) {
		final int start = toAppendTo.length();
		final String o = switchable.get((String) obj);
		toAppendTo.append(o);
		pos.setBeginIndex(start);
		pos.setEndIndex(toAppendTo.length());
		return toAppendTo;
	}

	@Override
	public Object parseObject(final String source, final ParsePosition pos) {
		for (final String key : switchable.keys()) {
			if (source.startsWith(key, pos.getIndex())) {
				pos.setIndex(pos.getIndex() + key.length());
				return switchable.get(key);
			}
		}
		final String o = switchable.getDefault();
		pos.setIndex(source.length());
		return o;
	}

	public Switchable<String, String> getSwitchable() {
		return switchable;
	}
}
