/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.NumberFormat;
import java.text.ParsePosition;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import woolpack.utils.CheckUtils;

/**
 * {@link DateFormat}の一覧を順にパースし、最初に成功した場合の値を返す{@link DateFormat}。
 * 適用しているパターン：Proxy。
 * 
 * @author nakamura
 * 
 */
public class TrysDateFormat extends DateFormat {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private Iterable<DateFormat> iterable;

	private void writeObject(final ObjectOutputStream out) throws IOException {
		final ArrayList<DateFormat> list = new ArrayList<DateFormat>();
		for (final DateFormat s : iterable) {
			list.add(s);
		}
		out.writeObject(list);
	}
	
	private void readObject(final ObjectInputStream in)
	throws IOException, ClassNotFoundException {
		final ArrayList<DateFormat> list = new ArrayList<DateFormat>();
		final Iterable it = (Iterable) in.readObject();
		for (final Object o : it) {
			list.add((DateFormat) o);
		}
		iterable = list;
	}
	
	/**
	 * コンストラクタ。
	 * 
	 * @param iterable
	 *            {@link DateFormat}の一覧。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 * @throws java.util.NoSuchElementException 引数が空の場合。
	 */
	public TrysDateFormat(final Iterable<DateFormat> iterable) {
		super();
		CheckUtils.checkNotNull(iterable);
		this.iterable = iterable;
		final DateFormat format = iterable.iterator().next();
		this.setCalendar((Calendar) format.getCalendar().clone());
		this.setNumberFormat((NumberFormat) format.getNumberFormat().clone());
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param array
	 *            {@link DateFormat}の一覧。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public TrysDateFormat(final DateFormat... array) {
		this(Arrays.asList(array));
	}

	@Override
	public StringBuffer format(final Date date, final StringBuffer toAppendTo,
			final FieldPosition fieldPosition) {
		return copy(iterable.iterator().next()).format(date, toAppendTo,
				fieldPosition);
	}

	@Override
	public Date parse(final String source, final ParsePosition pos) {
		int errorIndex = pos.getErrorIndex();
		for (final DateFormat f : iterable) {
			final ParsePosition pp = new ParsePosition(0);
			pp.setIndex(pos.getIndex());
			pp.setErrorIndex(pos.getErrorIndex());
			final Date d = copy(f).parse(source, pp);
			if (d != null) {
				pos.setIndex(pp.getIndex());
				return d;
			}
			errorIndex = pp.getErrorIndex();
		}
		pos.setErrorIndex(errorIndex);
		return null;
	}

	private static DateFormat copy(final DateFormat f) {
		return (DateFormat) f.clone();
	}

	public List<DateFormat> getIterable() {
		final List<DateFormat> list = new ArrayList<DateFormat>();
		for (final DateFormat f : iterable) {
			list.add(copy(f));
		}
		return Collections.unmodifiableList(list);
	}
}
