/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;

/**
 * 委譲先{@link InputStreamFactory}で{@link InputStream}
 * を生成して{@link Reader}に変換する{@link ReaderFactory}。
 * 適用しているパターン：Adapter。
 * 
 * @author nakamura
 * 
 */
public class InputStreamReaderFactory implements ReaderFactory {
	private final InputStreamFactory inputStreamFactory;

	private final String charset;

	/**
	 * コンストラクタ。
	 * 
	 * @param inputStreamFactory
	 *            {@link InputStream}を生成するための委譲先。
	 * @param charset
	 *            文字セット。
	 * @throws NullPointerException
	 *             inputStreamFactory が null の場合。
	 * @throws IllegalArgumentException (
	 *             {@link UnsupportedEncodingException})文字セットが不正な場合。
	 */
	public InputStreamReaderFactory(
			final InputStreamFactory inputStreamFactory, final String charset) {
		CheckUtils.checkNotNull(inputStreamFactory);
		this.inputStreamFactory = inputStreamFactory;
		this.charset = charset;

		if (charset != null && charset.length() > 0) {
			try {
				"a".getBytes(charset);
			} catch (final UnsupportedEncodingException e) {
				throw new IllegalArgumentException(e);
			}
		}
	}

	public Reader newInstance(final String id) throws IOException {
		final InputStream stream = inputStreamFactory.newInstance(id);
		if (stream == null) {
			return null;
		} else {
			try {
				if (charset == null || charset.length() == 0) {
					return new BufferedReader(new InputStreamReader(stream));
				} else {
					return new BufferedReader(new InputStreamReader(stream,
							charset));
				}
			} catch (final IOException e) {
				stream.close();
				throw e;
			}
		}
	}
}
