/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.utils.concurrent;

import java.util.concurrent.Semaphore;

import woolpack.utils.CheckUtils;

/**
 * {@link #acquire()}で{@link Semaphore#acquire()}を実行し、
 * {@link #release()}で{@link Semaphore#release()}を実行する{@link Acquirable}。
 * 適用しているパターン：Adapter, Template Method。
 * 
 * @author nakamura
 * 
 */
public class DoSemaphore implements Acquirable {
	private final Semaphore semaphore;

	/**
	 * コンストラクタ。
	 * 
	 * @param semaphore
	 *            委譲先。本クラスはこの引数の状態を変化させる。
	 * @throws NullPointerException
	 *             引数が null の場合。
	 */
	public DoSemaphore(final Semaphore semaphore) {
		CheckUtils.checkNotNull(semaphore);
		this.semaphore = semaphore;
	}

	public boolean acquire() {
		try {
			semaphore.acquire();
		} catch (final InterruptedException e) {
			processInterruptedException(e);
		}
		return true;
	}

	public void release() {
		semaphore.release();
	}

	/**
	 * {@link #acquire()}で{@link InterruptedException}
	 * が発生した場合に{@link #acquire()}から呼び出される(called)。
	 * 必要に応じてオーバライドすることができる。
	 * デフォルトは現在のスレッドに対して{@link Thread#interrupt()}を実行した後、
	 * 原因が引数である{@link IllegalStateException}を投げる。
	 * 
	 * @param e
	 *            原因。
	 * @throws IllegalStateException (
	 *             {@link InterruptedException})デフォルト実装では常に投げる。
	 */
	public void processInterruptedException(final InterruptedException e) {
		Thread.currentThread().interrupt();
		throw new IllegalStateException(e);
	}

	public Semaphore getSemaphore() {
		return semaphore;
	}
}
