/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor;

import java.util.Map;
import java.util.Map.Entry;

import woolpack.utils.Switchable;

/**
 * 定数と静的メソッドの集まり。
 * 
 * @author nakamura
 * 
 */
public final class HtmlVisitorConstants {

	/**
	 * {@link Switchable}を HTML dl タグで表現する{@link Acceptable}。
	 */
	public static final Acceptable<Switchable> SWITCH = DomVisitorConstants
			.addElementAcceptable("dl", new Acceptable<Switchable>() {
				private Acceptable<Object> keyAcceptable = DomVisitorConstants
						.addElementAcceptable("dt", VisitorConstants.ECHO);

				private Acceptable<Object> valueAcceptable = DomVisitorConstants
						.addElementAcceptable("dd", VisitorConstants.ECHO);

				public void accept(final Visitor visitor, final Switchable v) {
					for (final Object e : v.keys()) {
						keyAcceptable.accept(visitor, e);
						valueAcceptable.accept(visitor, v.get(e));
					}
					if (v.getDefault() != null) {
						keyAcceptable.accept(visitor, "デフォルト");
						valueAcceptable.accept(visitor, v.getDefault());
					}
				}
			});

	/**
	 * {@link Iterable}の内容を HTML ol タグで表現をする{@link Acceptable}。
	 */
	public static final Acceptable<Iterable> ITERABLE = DomVisitorConstants
			.addElementAcceptable("ol", new Acceptable<Iterable>() {
				private Acceptable<Object> acceptable = DomVisitorConstants
						.addElementAcceptable("li", VisitorConstants.ECHO);

				public void accept(final Visitor visitor, final Iterable v) {
					for (final Object e : v) {
						acceptable.accept(visitor, e);
					}
				}
			});

	/**
	 * HTML div タグを追加して {@link Acceptable#accept(Visitor, Object)}引数の V を
	 * {@link Visitor#visit(Object)}に渡す{@link Acceptable}。
	 */
	public static final Acceptable<Object> BLOCK = DomVisitorConstants
			.addElementAcceptable("div", VisitorConstants.ECHO);

	/**
	 * {@link Map}を HTML dl タグで表現する{@link Acceptable}。
	 */
	public static final Acceptable<Map> MAP = DomVisitorConstants
			.addElementAcceptable("dl", new Acceptable<Map>() {
				private Acceptable<Object> keyAcceptable = DomVisitorConstants
						.addElementAcceptable("dt", VisitorConstants.ECHO);

				private Acceptable<Object> valueAcceptable = DomVisitorConstants
						.addElementAcceptable("dd", VisitorConstants.ECHO);

				public void accept(final Visitor visitor, final Map v) {
					for (final Object entryObject : v.entrySet()) {
						final Entry entry = (Entry) entryObject;
						keyAcceptable.accept(visitor, entry.getKey());
						valueAcceptable.accept(visitor, entry.getValue());
					}
				}
			});

	private HtmlVisitorConstants() {
	} // カバレージがここを通過してはいけない
}
