/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.dom;

import java.util.Arrays;
import java.util.HashMap;

import junit.framework.TestCase;
import woolpack.adapter.JXP;
import woolpack.el.FixEL;
import woolpack.test.TestUtils;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.UtilsConstants;

public class DomConstantsTest extends TestCase {

	public void testEvalId() {
		final DomContext context = new DomContext();
		context.setId("id0");
		assertEquals("id0", DomConstants.EVAL_ID.eval(context));
		context.setId("");
		assertEquals("", DomConstants.EVAL_ID.eval(context));
		context.setId(null);
		assertEquals(null, DomConstants.EVAL_ID.eval(context));
	}

	public void testEvalNodeName() {
		final DomExpression expression = new Serial(
				TestUtils
						.getToNodeBody("<TEXTAREA name=\"name0\" ></TEXTAREA>"
								+ "<INPUT name=\"name1\" />"
								+ "<SELECT name=\"name2\" ><OPTION value=\"0\"/>zero</SELECT>"),
				new XPath(new JXP("//*[@name]"), new Branch<String>(
						DomConstants.EVAL_NODE_NAME,
						new SwitchBuilder<String, DomExpression>().put(
								"TEXTAREA",
								new UpdateAttrValue("name", new FixEL(
										"newName0"))).put(
								"INPUT",
								new UpdateAttrValue("name", new FixEL(
										"newName1"))).get())));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils
				.equalsBody(
						context,
						"<TEXTAREA name=\"newName0\" ></TEXTAREA>"
								+ "<INPUT name=\"newName1\" />"
								+ "<SELECT name=\"name2\" ><OPTION value=\"0\"/>zero</SELECT>"));
	}

	public void testEl() {
		final DomContext context0 = new DomContext();
		context0.setId("id0");
		context0.setConfig(new HashMap<String, Object>());
		context0.setContainer(new HashMap());
		context0.setInput(new HashMap());
		context0.setRequest(new HashMap<String, Object>());
		context0.setSession(UtilsConstants.concurrentMap(
				new HashMap<String, Object>(), new Object()));
		context0.setApplication(UtilsConstants.concurrentMap(
				new HashMap<String, Object>(), new Object()));

		assertEquals(context0.getConfig(), DomConstants.CONFIG_EL
				.getValue(context0));
		assertEquals(context0.getContainer(), DomConstants.CONTAINER_EL
				.getValue(context0));
		assertEquals(context0.getInput(), DomConstants.INPUT_EL
				.getValue(context0));
		assertEquals(context0.getRequest(), DomConstants.REQUEST_EL
				.getValue(context0));
		assertEquals(context0.getSession(), DomConstants.SESSION_EL
				.getValue(context0));
		assertEquals(context0.getApplication(), DomConstants.APPLICATION_EL
				.getValue(context0));
		assertEquals(context0.getLocal(), DomConstants.LOCAL_EL
				.getValue(context0));
	}

	public void testNull() {
		final DomContext context = new DomContext();
		DomConstants.NULL.interpret(context);
	}

	public void testRemove() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeBody("<SPAN id=\"id0\" >value0</SPAN>"), new XPath(
				new JXP("id", "id0"), DomConstants.REMOVE_THIS));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, ""));
	}

	public void testRemoveChildren() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeBody("<SPAN id=\"id0\" >value0</SPAN>"), new XPath(
				new JXP("id", "id0"), DomConstants.REMOVE_CHILDREN));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<SPAN id=\"id0\" />"));
	}

	public void testRetainChildren() {
		final DomExpression expression = new Serial(
				Arrays
						.asList(
								TestUtils
										.getToNodeBody("<SPAN id=\"id0\" >hoge0<SPAN id=\"id1\" >hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1</SPAN>piyo0</SPAN>"),
								new XPath(new JXP("id", "id1"),
										DomConstants.RETAIN_CHILDREN)));
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils
				.equalsBody(context,
						"<SPAN id=\"id0\" >hoge0hoge1<SPAN id=\"id2\" >value0</SPAN>piyo1piyo0</SPAN>"));
	}

	public void testGetDocumentNode() {
		final DomExpression expression = TestUtils
				.getToNodeForm("<INPUT type=\"text\" name=\"name0\" value=\"value0\" />");
		final DomContext context = new DomContext();
		expression.interpret(context);
		assertEquals(context.getNode(), DomConstants.getDocumentNode(context
				.getNode()));
		assertEquals(context.getNode(), DomConstants.getDocumentNode(context
				.getNode().getFirstChild()));
	}

	public void testClearSession() {
		final DomContext context = new DomContext();
		context.setSession(UtilsConstants.concurrentMap(
				new HashMap<String, Object>(), new Object()));
		context.getSession().put("key0", "value0");
		context.getSession().put("key1", "value1");
		assertFalse(context.getSession().isEmpty());
		DomConstants.CLEAR_SESSION.interpret(context);
		assertTrue(context.getSession().isEmpty());
	}
}
