/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.ee;

import java.util.Arrays;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;

import junit.framework.TestCase;

import woolpack.LoadAllTests;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.test.TestConcurrent;
import woolpack.test.ee.EETestUtils;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.concurrent.Acquirable;
import woolpack.utils.concurrent.TrySemaphore;

public class DoAcquireTransactionLoadTest extends TestCase {

	private void scenario(final int testCount, final int threshold)
			throws InterruptedException {
		final Object lock = new Object();
		final TestConcurrent concurrentTrue0 = new TestConcurrent("true0",
				lock, 1, LoadAllTests.MAX_CONCURRENT, LoadAllTests.MAX_THREAD,
				threshold);
		final TestConcurrent concurrentFalse0 = new TestConcurrent("false0",
				lock, 1, LoadAllTests.MAX_THREAD - LoadAllTests.MAX_CONCURRENT,
				LoadAllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentRelease0 = new TestConcurrent(
				"release0", lock, 1, LoadAllTests.MAX_CONCURRENT,
				LoadAllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentTrue1 = new TestConcurrent("true1",
				lock, 1, LoadAllTests.MAX_CONCURRENT, LoadAllTests.MAX_THREAD,
				threshold);
		final TestConcurrent concurrentFalse1 = new TestConcurrent("false1",
				lock, 1, LoadAllTests.MAX_THREAD, LoadAllTests.MAX_THREAD,
				threshold);
		final TestConcurrent concurrentRelease1 = new TestConcurrent(
				"release1", lock, 1, LoadAllTests.MAX_CONCURRENT,
				LoadAllTests.MAX_THREAD, threshold);
		final TestConcurrent concurrentTrue2 = new TestConcurrent("true2",
				lock, 1, LoadAllTests.MAX_CONCURRENT, LoadAllTests.MAX_THREAD,
				threshold);
		final TestConcurrent concurrentFalse2 = new TestConcurrent("false2",
				lock, 1, LoadAllTests.MAX_THREAD, LoadAllTests.MAX_THREAD,
				threshold);
		final TestConcurrent concurrentRelease2 = new TestConcurrent(
				"release2", lock, 1, LoadAllTests.MAX_CONCURRENT,
				LoadAllTests.MAX_THREAD, threshold);

		final DomExpression expression = new DoAcquireTransaction(
				new SwitchBuilder<String, Acquirable>().put(
						"id0",
						new TrySemaphore(new Semaphore(
								LoadAllTests.MAX_CONCURRENT, true))).get(),
				Arrays.asList("id1"), Arrays.asList("id2"),
				new DomExpression() {
					public void interpret(final DomContext context) {
						if ("id0".equals(context.getId())) {
							concurrentTrue0.execute();
						} else if ("id1".equals(context.getId())) {
							concurrentTrue1.execute();
						} else if ("id2".equals(context.getId())) {
							concurrentTrue2.execute();
						}
					}
				}, new DomExpression() {
					public void interpret(final DomContext context) {
						if ("id0".equals(context.getId())) {
							concurrentFalse0.execute();
						} else if ("id1".equals(context.getId())) {
							concurrentFalse1.execute();
						} else if ("id2".equals(context.getId())) {
							concurrentFalse2.execute();
						}
					}
				}, new DomExpression() {
					public void interpret(final DomContext context) {
						if ("id0".equals(context.getId())) {
							concurrentRelease0.execute();
						} else if ("id1".equals(context.getId())) {
							concurrentRelease1.execute();
						} else if ("id2".equals(context.getId())) {
							concurrentRelease2.execute();
						}
					}
				});

		final CountDownLatch latch = new CountDownLatch(LoadAllTests.MAX_THREAD);
		for (int i = 0; i < LoadAllTests.MAX_THREAD; i++) {
			final int ii = i;
			LoadAllTests.THREAD_POOL.execute(new Runnable() {
				public void run() {
					try {
						concurrentTrue0.setThreadId(ii);
						concurrentFalse0.setThreadId(ii);
						concurrentRelease0.setThreadId(ii);
						concurrentTrue1.setThreadId(ii);
						concurrentFalse1.setThreadId(ii);
						concurrentRelease1.setThreadId(ii);
						concurrentTrue2.setThreadId(ii);
						concurrentFalse2.setThreadId(ii);
						concurrentRelease2.setThreadId(ii);
						final ConcurrentMap<String, Object> session0 = EETestUtils
								.generateSession();
						for (int i = 0; i < testCount; i++) {
							Thread.yield();
							final DomContext context = new DomContext();
							context.setSession(session0);
							context.setId("id0");
							try {
								expression.interpret(context);
							} catch (final IllegalStateException e) {
							}

							Thread.yield();
							context.setId("id1");
							try {
								expression.interpret(context);
							} catch (final IllegalStateException e) {
							}

							Thread.yield();
							context.setId("id2");
							try {
								expression.interpret(context);
							} catch (final IllegalStateException e) {
							}
							Thread.yield();
						}
					} finally {
						latch.countDown();
					}
				}
			});
		}
		latch.await();

		// System.out.println("-- TrySemaphoreTransactionTest start --");
		// concurrentTrue0.print();
		// concurrentFalse0.print();
		// concurrentRelease0.print();
		// concurrentTrue1.print();
		// concurrentFalse1.print();
		// concurrentRelease1.print();
		// concurrentTrue2.print();
		// concurrentFalse2.print();
		// concurrentRelease2.print();
		// System.out.println("-- TrySemaphoreTransactionTest end --");
		assertTrue(concurrentTrue0.assertValid());
		assertTrue(concurrentFalse0.assertValid());
		assertTrue(concurrentRelease0.assertValid());
		assertTrue(concurrentTrue1.assertValid());
		assertTrue(concurrentFalse1.assertValid());
		assertTrue(concurrentRelease1.assertValid());
		assertTrue(concurrentTrue2.assertValid());
		assertTrue(concurrentFalse2.assertValid());
		assertTrue(concurrentRelease2.assertValid());
	}

	public void testNormal() throws InterruptedException {
		scenario(LoadAllTests.TEST_COUNT, LoadAllTests.TEST_THRESHOLD);
	}
}
