/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;
import woolpack.adapter.JXPFactory;
import woolpack.dom.DomConstants;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Serial;
import woolpack.el.ELConstants;
import woolpack.test.TestUtils;
import woolpack.utils.MapBuilder;
import woolpack.utils.UtilsConstants;

public class AutoUpdateTest extends TestCase {

	public void testConstructor() {
		try {
			new AutoUpdate(null, ELConstants.NULL, UtilsConstants.ATOM_SET,
					ELConstants.NULL, new JXPFactory());
			fail();
		} catch (final NullPointerException expected) {
		}
		try {
			new AutoUpdate(Arrays.asList("id"), null, UtilsConstants.ATOM_SET,
					ELConstants.NULL, new JXPFactory());
			fail();
		} catch (final NullPointerException expected) {
		}
		try {
			new AutoUpdate(Arrays.asList("id"), ELConstants.NULL, null,
					ELConstants.NULL, new JXPFactory());
			fail();
		} catch (final NullPointerException expected) {
		}
		try {
			new AutoUpdate(Arrays.asList("id"), ELConstants.NULL,
					UtilsConstants.ATOM_SET, null, new JXPFactory());
			fail();
		} catch (final NullPointerException expected) {
		}
		try {
			new AutoUpdate(Arrays.asList("id"), ELConstants.NULL,
					UtilsConstants.ATOM_SET, ELConstants.NULL, null);
			fail();
		} catch (final NullPointerException expected) {
		}
	}

	public void testSimple() {
		final DomExpression expression = new Serial(
				TestUtils
						.getToNodeForm("<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"
								+ "<DIV id=\"div0\" >value0</DIV>"
								+ "<SPAN id=\"span0\" >value0</SPAN>"
								+ "<INPUT name=\"input0\" value=\"value0\" />"
								+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"),
				new AutoUpdate(Arrays.asList("name", "id"),
						DomConstants.CONTAINER_EL, new JXPFactory()));
		final DomContext context = new DomContext();
		context.setContainer(MapBuilder.get(new HashMap<String, Object>()).put(
				"textarea0", "textareavalue0").put("div0", "divvalue0").put(
				"span0", "spanvalue0").put("input0", "inputvalue0").get());
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context,
				"<TEXTAREA name=\"textarea0\" >textareavalue0</TEXTAREA>"
						+ "<DIV id=\"div0\" >divvalue0</DIV>"
						+ "<SPAN id=\"span0\" >spanvalue0</SPAN>"
						+ "<INPUT name=\"input0\" value=\"inputvalue0\" />"
						+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"));
	}

	public void testIterate() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeBody("<TABLE>"
						+ "<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>"
						+ "</TABLE>"), new AutoUpdate(Arrays.asList("name",
				"id"), DomConstants.LOCAL_EL, new JXPFactory()));
		final DomContext context = new DomContext();
		context.getLocal().put(
				"iterate0",
				Arrays.asList(MapBuilder.get(new HashMap<String, Object>())
						.put("id0", "value0").get(), MapBuilder.get(
						new HashMap<String, Object>()).put("id0", "value1")
						.get()));
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<TABLE>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value0</TD></TR>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value1</TD></TR>"
				+ "</TABLE>"));
	}

	public void testIterateEmpty() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeBody("<TABLE>"
						+ "<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>"
						+ "</TABLE>"), new AutoUpdate(Arrays.asList("name",
				"id"), DomConstants.LOCAL_EL, new JXPFactory()));
		final DomContext context = new DomContext();
		context.getLocal().put("iterate0", new HashMap[] {});
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<TABLE></TABLE>"));
	}

	public void testArray() {
		final DomExpression expression = new Serial(TestUtils
				.getToNodeBody("<TABLE>"
						+ "<TR id=\"iterate0\"><TD id=\"id0\">dummy</TD></TR>"
						+ "</TABLE>"), new AutoUpdate(Arrays.asList("name",
				"id"), DomConstants.LOCAL_EL, new JXPFactory()));
		final DomContext context = new DomContext();
		context.getLocal().put(
				"iterate0",
				new Map[] {
						MapBuilder.get(new HashMap<String, Object>()).put(
								"id0", "value0").get(),
						MapBuilder.get(new HashMap<String, Object>()).put(
								"id0", "value1").get(), });
		expression.interpret(context);

		assertTrue(TestUtils.equalsBody(context, "<TABLE>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value0</TD></TR>"
				+ "<TR id=\"iterate0\"><TD id=\"id0\">value1</TD></TR>"
				+ "</TABLE>"));
	}

	public void testNotFound() {
		final DomExpression expression = new Serial(
				TestUtils
						.getToNodeForm("<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"
								+ "<DIV id=\"div0\" >value0</DIV>"
								+ "<SPAN id=\"span0\" >value0</SPAN>"
								+ "<INPUT name=\"input0\" value=\"value0\" />"
								+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"),
				new AutoUpdate(Arrays.asList("name", "id"),
						DomConstants.CONTAINER_EL, new JXPFactory()));
		final DomContext context = new DomContext();
		context.setContainer(new Object());
		expression.interpret(context);

		assertTrue(TestUtils.equalsForm(context,
				"<TEXTAREA name=\"textarea0\" >value0</TEXTAREA>"
						+ "<DIV id=\"div0\" >value0</DIV>"
						+ "<SPAN id=\"span0\" >value0</SPAN>"
						+ "<INPUT name=\"input0\" value=\"value0\" />"
						+ "<SPAN id=\"span1\" >spanvalue1</SPAN>"));
	}
}
