/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import junit.framework.TestCase;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import woolpack.adapter.JXP;
import woolpack.dom.DomContext;
import woolpack.dom.DomExpression;
import woolpack.dom.Serial;
import woolpack.test.TestUtils;

public class HtmlConstantsTest extends TestCase {

	public void testNormalizeCase() {
		final DomExpression expression = new Serial(
				TestUtils
						.getToNode("<html><BODY>"
								+ "<sPaN attra=\"valuea\" ATTRB=\"VALUEB\" attrC=\"valueC\" attrd=\"valued\" ATTRE=\"VALUEE\" attrF=\"valueF\">aBcD</sPaN>"
								+ "<SPAN attrg=\"valueg\" ATTRH=\"VALUEH\" attrI=\"valueI\" attrj=\"valuej\" ATTRK=\"VALUEK\" attrL=\"valueL\">aBcD</SPAN>"
								+ "</BODY></html>"),
				HtmlConstants.NORMALIZE_CASE);
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils
				.equals(
						context,
						"<HTML><BODY>"
								+ "<SPAN attra=\"valuea\" attrb=\"VALUEB\" attrc=\"valueC\" attrd=\"valued\" attre=\"VALUEE\" attrf=\"valueF\">aBcD</SPAN>"
								+ "<SPAN attrg=\"valueg\" attrh=\"VALUEH\" attri=\"valueI\" attrj=\"valuej\" attrk=\"VALUEK\" attrl=\"valueL\">aBcD</SPAN>"
								+ "</BODY></HTML>"));
	}

	public void testNormalize() {
		final DomExpression expression = TestUtils
				.getToNode("<HTML><BODY>a</BODY></HTML>");
		final DomContext context = new DomContext();
		expression.interpret(context);
		final Node n0 = context.getNode();
		final Node n1 = new JXP("/HTML/BODY").evaluateOne(n0);
		n1.appendChild(((Document) n0).createTextNode("b"));
		n1.appendChild(((Document) n0).createTextNode("c"));
		assertEquals(3, n1.getChildNodes().getLength());
		HtmlConstants.NORMALIZE.interpret(context);
		assertEquals(1, n1.getChildNodes().getLength());
	}

	public void testRemoveComment() {
		final DomExpression expression = new Serial(
				TestUtils
						.getToNode("<HTML><BODY><SCRIPT><!-- var a = 1;  --></SCRIPT>hoge<!-- naru -->piyo</BODY></HTML>"),
				HtmlConstants.REMOVE_COMMENT);
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils
				.equals(context,
						"<HTML><BODY><SCRIPT><!-- var a = 1;  --></SCRIPT>hogepiyo</BODY></HTML>"));
	}

	public void testCompressSpace() {
		final DomExpression expression = new Serial(
				TestUtils
						.getToNode("<HTML><BODY><SCRIPT><!--   --></SCRIPT> \t\r\nhoge\t\r\n piyo\r\n \tnaru\n  \t\t</BODY></HTML>"),
				HtmlConstants.COMPRESS_SPACE);
		final DomContext context = new DomContext();
		expression.interpret(context);

		assertTrue(TestUtils
				.equals(context,
						"<HTML><BODY><SCRIPT><!--   --></SCRIPT> hoge piyo naru </BODY></HTML>"));
	}
}
