/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.text;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.Format;
import java.text.ParseException;

import junit.framework.TestCase;

public class ToBigDecimalFormatTest extends TestCase {

	public void testConstructor() {
		try {
			new ToBigDecimalFormat(null);
			fail();
		} catch (final NullPointerException e) {
		}
	}

	private void scenarioParseBigDecimalTrue(final Format format)
			throws ParseException {
		assertEquals("1", format.format(new BigDecimal(1)));
		assertEquals(new BigDecimal(1), format.parseObject("1"));
		{
			final BigDecimal b = new BigDecimal(Double.MAX_VALUE)
					.add(new BigDecimal(1));
			assertEquals(b, format.parseObject(b.toPlainString()));
		}
		{
			final BigDecimal b = new BigDecimal(Double.MIN_VALUE)
					.subtract(new BigDecimal(1));
			assertEquals(b, format.parseObject(b.toPlainString()));

		}
		assertEquals(new BigDecimal(1.5), format.parseObject("1.5"));

		try {
			format.parseObject("a");
			fail();
		} catch (final ParseException e) {
		}
	}

	private void scenarioParseBigDecimalFalse(final Format format)
			throws ParseException {
		assertEquals("1", format.format(new BigDecimal(1)));
		assertEquals(new BigDecimal(1), format.parseObject("1"));
		{
			final BigDecimal a = new BigDecimal(Double.MAX_VALUE);
			final BigDecimal b = a.add(new BigDecimal(1));
			assertEquals(a, format.parseObject(b.toPlainString()));
		}
		{
			final BigDecimal a = new BigDecimal(Double.MIN_VALUE);
			// Double にとっては 0-1 == -1
			final BigDecimal b = a.subtract(new BigDecimal(1));
			assertEquals(new BigDecimal(-1), format.parseObject(b
					.toPlainString()));

		}
		assertEquals(new BigDecimal(1.5), format.parseObject("1.5"));

		try {
			format.parseObject("a");
			fail();
		} catch (final ParseException e) {
		}
	}

	public void testNormal() throws ParseException {
		final Format format0 = new ToBigDecimalFormat();
		scenarioParseBigDecimalTrue(format0);
		final Format format1 = (Format) format0.clone();
		assertTrue(format0 != format1);
		scenarioParseBigDecimalTrue(format1);
	}

	public void testNormal2() throws ParseException {
		final DecimalFormat base = new DecimalFormat();
		base.setParseBigDecimal(true);
		final Format format0 = new ToBigDecimalFormat(base);
		scenarioParseBigDecimalTrue(format0);
		final Format format1 = (Format) format0.clone();
		assertTrue(format0 != format1);
		scenarioParseBigDecimalTrue(format1);
	}

	public void testNormal3() throws ParseException {
		final DecimalFormat base = new DecimalFormat();
		base.setParseBigDecimal(false);
		final Format format0 = new ToBigDecimalFormat(base);
		scenarioParseBigDecimalFalse(format0);
		final Format format1 = (Format) format0.clone();
		assertTrue(format0 != format1);
		scenarioParseBigDecimalFalse(format1);
	}
}
