/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import junit.framework.TestCase;
import woolpack.test.TestUtils;
import woolpack.text.RegExpFormat;
import woolpack.utils.MapBuilder;

public class ValueLoopValidatorTest extends TestCase {

	public void testConstructor() {
		try {
			new ValueLoopValidator(null, ValidatorConstants.FALSE);
			fail();
		} catch (final NullPointerException expected) {
		}
		try {
			new ValueLoopValidator(ValidatorConstants.ANDAND, null);
			fail();
		} catch (final NullPointerException expected) {
		}
	}

	public void testNull() {
		final CountValidator counter0 = new CountValidator(
				ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(
				ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(new HashMap<String, List<Object>>());
		context.setTmpKey("key0");

		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getCount());
		assertEquals("key0", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
	}

	public void testObject() {
		final CountValidator counter0 = new CountValidator(
				ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(
				ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(new HashMap<String, List<Object>>());
		context.setTmpKey("key0");

		context.setTmpValue(new Object());
		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getCount());
		assertEquals("key0", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
	}

	public void testFalse() {
		final CountValidator counter0 = new CountValidator(
				ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(
				ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String, List<Object>>())
				.put("key0", Arrays.asList((Object) "a", "b", "c")).get());

		assertFalse(expression.interpret(context));
		assertEquals(1, counter0.getCount());
		assertEquals("key0", counter0.getContextList().getLast().getTmpKey());
		assertEquals(0, counter0.getContextList().getLast().getTmpIndex());
	}

	public void testTrue() {
		final CountValidator counter0 = new CountValidator(
				ValidatorConstants.TRUE);
		final ValidatorExpression expression = new ValueLoopValidator(
				ValidatorConstants.ANDAND, counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String, List<Object>>())
				.put("key0", Arrays.asList((Object) "a", "b", "c")).get());

		assertTrue(expression.interpret(context));
		assertEquals(3, counter0.getCount());
		{
			final ValidatorContext tmpContext = counter0.getContextList()
					.removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(2, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList()
					.removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(1, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList()
					.removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(0, tmpContext.getTmpIndex());
		}
	}

	public void testParse() {
		final ValidatorExpression expression = new ValueLoopValidator(
				ValidatorConstants.ANDAND, new ParseValidator(new RegExpFormat(
						"[ab]", "\\*")));

		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String, List<Object>>())
				.put("key0", Arrays.asList((Object) "a", "b", "c")).get());

		assertTrue(expression.interpret(context));
		assertTrue(TestUtils.equals(MapBuilder.get(
				new HashMap<String, Object>()).put("key0",
				Arrays.asList("*", "*", "c")).get(), context.getInputMap()));
	}

	public void testSimpleConstractor() {
		final CountValidator counter0 = new CountValidator(
				ValidatorConstants.FALSE);
		final ValidatorExpression expression = new ValueLoopValidator(counter0);
		final ValidatorContext context = new ValidatorContext();
		context.setTmpKey("key0");
		context.setInputMap(MapBuilder.get(new HashMap<String, List<Object>>())
				.put("key0", Arrays.asList((Object) "a", "b", "c")).get());

		assertFalse(expression.interpret(context));
		assertEquals(3, counter0.getCount());
		{
			final ValidatorContext tmpContext = counter0.getContextList()
					.removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(2, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList()
					.removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(1, tmpContext.getTmpIndex());
		}
		{
			final ValidatorContext tmpContext = counter0.getContextList()
					.removeLast();
			assertEquals("key0", tmpContext.getTmpKey());
			assertEquals(0, tmpContext.getTmpIndex());
		}
	}
}
