/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.Map;

import junit.framework.TestCase;

import woolpack.test.TestBean;
import woolpack.test.TestUtils;
import woolpack.utils.MapBuilder;
import woolpack.utils.SwitchBuilder;
import woolpack.utils.Switchable;

public class VisitorConstantsTest extends TestCase {

	public void testIterable() {
		final CountAcceptor<Object> counter = new CountAcceptor<Object>();
		final Visitor visitor = new Visitor();
		visitor.setMap(MapBuilder.get(new LinkedHashMap<Object, Acceptable>())
				.put(Iterable.class, VisitorConstants.ITERABLE).put(
						Object.class, counter).get());
		visitor.visit(Arrays.asList("value0", "value1", "value2"));
		assertTrue(TestUtils.equals(
				Arrays.asList("value0", "value1", "value2"), counter.getList()));
	}

	public void testMap() {
		final CountAcceptor<Object> counter = new CountAcceptor<Object>();
		final Visitor visitor = new Visitor();
		visitor.setMap(MapBuilder.get(new LinkedHashMap<Object, Acceptable>())
				.put(Map.class, VisitorConstants.MAP)
				.put(Object.class, counter).get());
		visitor.visit(MapBuilder.get(new LinkedHashMap<Object, Object>()).put(
				"key0", "value0").put("key1", "value1").get());
		assertTrue(TestUtils.equals(Arrays.asList("key0", "value0", "key1",
				"value1"), counter.getList()));
	}

	public void testReflection() {
		final CountAcceptor<Object> counter = new CountAcceptor<Object>();
		final Visitor visitor = new Visitor();
		visitor.setMap(MapBuilder.get(new LinkedHashMap<Object, Acceptable>())
				.put(TestBean.class, VisitorConstants.REFLECTION).put(
						Object.class, counter).get());
		final TestBean bean = new TestBean();
		bean.setMyCollection(Arrays.asList("c0", "c1"));
		bean.setMyInt(3);
		bean.setMyIntArray(new int[] { 5, 7 });
		visitor.visit(bean);
		assertTrue(TestUtils.equals(Arrays.asList("myCollection", Arrays
				.asList("c0", "c1"), "myInt", Integer.valueOf(3), "myIntArray",
				bean.getMyIntArray()), counter.getList()));
	}

	public void testSwitchable() {
		final CountAcceptor<Object> counter = new CountAcceptor<Object>();
		final Visitor visitor = new Visitor();
		visitor.setMap(MapBuilder.get(new LinkedHashMap<Object, Acceptable>())
				.put(Switchable.class,
						VisitorConstants.switchableAcceptable("defaultKey"))
				.put(Object.class, counter).get());
		visitor.visit(new SwitchBuilder<Object, Object>().put("key0", "value0")
				.put("key1", "value1").get("defaultValue"));
		assertTrue(TestUtils.equals(Arrays.asList("key0", "value0", "key1",
				"value1", "defaultKey", "defaultValue"), counter.getList()));
	}

	public void testSerial() {
		final CountAcceptor<Object> counter = new CountAcceptor<Object>();
		final Visitor visitor = new Visitor();
		visitor.setMap(MapBuilder.get(new LinkedHashMap<Object, Acceptable>())
				.put(String.class, counter).put(
						Object.class,
						VisitorConstants.serialAcceptable(VisitorConstants
								.visitAcceptable("a"), VisitorConstants
								.visitAcceptable("b"))).get());
		visitor.visit(Integer.valueOf(2));
		assertTrue(TestUtils.equals(Arrays.asList("a", "b"), counter.getList()));
	}
}
