/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.id;

import woolpack.fn.Fn;

/**
 * 識別子のコンテキストに関するユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 *
 */
public final class IdUtils {
	
	/**
	 * {@link IdContext#getId()}を返す関数です。
	 */
	public static final Fn<IdContext, String, RuntimeException> GET_ID = new IdGetter<RuntimeException>();
	
	private IdUtils() {
	}
	
	/**
	 * {@link IdContext#getId()}を返す関数を生成します。
	 * @param <E>
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<IdContext, String, E> getId() {
		return new IdGetter<E>();
	}
	
	/**
	 * id の変換を委譲する関数を生成します。
	 * @param <R>
	 * @param <E>
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <R, E extends Exception> Fn<IdContext, R, E> convertId(
			final Fn<? super String, String, ? extends E> fn) {
		return new IdConverter<R, E>(fn);
	}
	
	/**
	 * 委譲先から復帰したときに識別子の値を呼び出し時の状態に初期化する{@link Fn}です。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param fn 委譲先。
	 * @return 関数。
	 */
	public static <C extends IdContext, R, E extends Exception> Fn<C, R, E> localId(
			final Fn<? super C, ? extends R, ? extends E> fn) {
		return new LocalId<C, R, E>(fn);
	}
}
