/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.Iterator;
import java.util.List;

import woolpack.bool.BooleanState;
import woolpack.fn.Delegator;
import woolpack.fn.Fn;

public class ValueLoopValidator<E extends Exception> extends Delegator<ValidatorContext, Boolean, E> {
	private Fn<Object, ? extends BooleanState, ? extends RuntimeException> operator;

	public ValueLoopValidator(
			final Fn<Object, ? extends BooleanState, ? extends RuntimeException> operator,
			final Fn<? super ValidatorContext, Boolean, ? extends E> fn) {
		super(fn);
		this.operator = operator;
	}

	@Override
	public Boolean exec(final ValidatorContext context) throws E {
		final int length;
		{
			final List<Object> list = context.getInputMap().get(context.getKey());
			if (list == null) {
				length = 1;
			} else {
				length = list.size();
			}
		}

		final Iterable<Fn<ValidatorContext, Boolean, ? extends E>> iterable = 
			new Iterable<Fn<ValidatorContext, Boolean, ? extends E>>() {
				public Iterator<Fn<ValidatorContext, Boolean, ? extends E>> iterator() {
					return new Iterator<Fn<ValidatorContext, Boolean, ? extends E>>() {
						private int i = 0;

						public boolean hasNext() {
							return i < length;
						}

						public Fn<ValidatorContext, Boolean, ? extends E> next() {
							return new Fn<ValidatorContext, Boolean, E>() {
								public Boolean exec(
										final ValidatorContext context) throws E {
									final int baseIndex = context.getIndex();
									try {
										context.setIndex(i++);
										return ValueLoopValidator.super.exec(context);
									} finally {
										context.setIndex(baseIndex);
									}
								}
							};
						}
						public void remove() {
							throw new UnsupportedOperationException();
						}
					};
				}
			};
		
		final BooleanState state = operator.exec(null);
		for (final Fn<? super ValidatorContext, Boolean, ? extends E> e : iterable) {
			state.in(e.exec(context));
			if (state.isStopped()) {
				break;
			}
		}
		return state.isCurrent();
	}

	public Fn<Object, ? extends BooleanState, ? extends RuntimeException> getOperator() {
		return operator;
	}
	public void setOperator(final Fn<Object, ? extends BooleanState, ? extends RuntimeException> operator) {
		this.operator = operator;
	}
}
