/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import javax.sql.DataSource;

import junit.framework.TestCase;
import woolpack.adapter.HSQLDBUtils;
import woolpack.fn.FnUtils;
import woolpack.sql.fn.SqlFnUtils;
import woolpack.sql.meta.SqlMetaUtils;
import woolpack.sql.meta.TableInfo;
import woolpack.sql.meta.TableInfoUtils;
import woolpack.sql.tx.TxBuilder;
import woolpack.sql.tx.SqlTxUtils;
import woolpack.utils.Utils;
import woolpack.visitor.Visitor;
import woolpack.visitor.VisitorAppUtils;

public class TableInfoUtilsTest extends TestCase {

	public void testGetTableInfoList() throws Exception {
		final List<String> list = Arrays.asList(
				"CREATE TABLE MY_TABLE (" +
				"MY_ID INTEGER, " +
				"MY_SEQ INTEGER, " +
				"MY_STRING VARCHAR, " +
				"MY_INT0 INTEGER, " +
				"MY_INT1 INTEGER)",
				
				"ALTER TABLE MY_TABLE ADD PRIMARY KEY (MY_ID, MY_SEQ)",
				
				"CREATE TABLE YOUR_TABLE (" +
				"MY_INT0 INTEGER, " +
				"MY_STRING VARCHAR)",
				
				"ALTER TABLE YOUR_TABLE ADD PRIMARY KEY (MY_INT0, MY_STRING)",
				
				"ALTER TABLE MY_TABLE ADD FOREIGN KEY (MY_INT0, MY_STRING) " +
				"REFERENCES YOUR_TABLE (MY_INT0, MY_STRING)",
				
				"CREATE TABLE HIS_TABLE (" +
				"MY_INT INTEGER, " +
				"HIS_STRING VARCHAR)",
				
				"ALTER TABLE HIS_TABLE ADD PRIMARY KEY (MY_INT)",
				
				"ALTER TABLE MY_TABLE ADD FOREIGN KEY (MY_INT0) " +
				"REFERENCES HIS_TABLE (MY_INT)",
				
				"ALTER TABLE MY_TABLE ADD FOREIGN KEY (MY_INT1) " +
				"REFERENCES HIS_TABLE (MY_INT)"
				);
		
		final DataSource dataSource = HSQLDBUtils.getDataSource(false);
		final TxBuilder builder = new TxBuilder(dataSource);
		for (final String query : list) {
			assertEquals(Integer.valueOf(0), SqlTxUtils.tx(builder.getTxDataSource(),
					SqlFnUtils.inputStatement(
							builder.getTmpDataSource(),
							query,
							SqlFnUtils.GET_COUNT,
							FnUtils.fix(null))
			).exec(null));
		}
		
		final List<TableInfo> result = TableInfoUtils.getTableInfoList(dataSource);
		
		assertEquals(
				new HashSet<String>(list),
				new HashSet<String>(SqlMetaUtils.generateCreateQuery(result)));
		
		final String resultString;
		{
			final Visitor<StringBuilder, RuntimeException> visitor = new Visitor<StringBuilder, RuntimeException>();
			visitor.setMap(VisitorAppUtils.DUMP);
			visitor.setSubContext(new StringBuilder());
			visitor.visit(result);
			resultString = visitor.getSubContext().toString();
		}
		System.out.println(resultString);
	}
	
	
	private static TableInfo getTableInfo() {
		final TableInfo tableInfo = new TableInfo();
		tableInfo.setTableName("my_table");
		tableInfo.setColNameList(Arrays.asList("my_id", "my_seq", "my_string", "my_int"));
		tableInfo.setPkNameList(Arrays.asList("my_id", "my_seq"));
		return tableInfo;
	}
	
	public void testConvertJavaNameTableInfoMap() {
		final TableInfo info0 = getTableInfo();
		final TableInfo info1 = getTableInfo();
		info1.setTableName("my_table1");
		assertEquals(
				Utils.map("my_table", info0).map("my_table1", info1),
				TableInfoUtils.convertToMap(Arrays.asList(info0, info1)));
	}
}
