/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import junit.framework.TestCase;
import woolpack.el.ArrayPathEL;
import woolpack.el.CachePropertyELFactory;
import woolpack.el.ELUtils;
import woolpack.el.FixEL;
import woolpack.el.GettingEL;
import woolpack.el.MapEL;
import woolpack.el.MapPropertyELFactory;
import woolpack.el.MapSimilarPropertyELFactory;
import woolpack.el.MaybeEL;
import woolpack.el.MaybePropertyELFactory;
import woolpack.el.PathEL;
import woolpack.el.PropertyEL;
import woolpack.el.ReflectionPropertyELFactory;
import woolpack.el.SimilarPropertyELFactory;
import woolpack.el.ThisEL;
import woolpack.fn.FnUtils;
import woolpack.sql.adapter.DataSourceAdapter;
import woolpack.sql.fn.PreparedStatementInfo;
import woolpack.sql.tx.TxBuilder;
import woolpack.typeconvert.ConvertContext;
import woolpack.typeconvert.Converter;
import woolpack.typeconvert.DelegationIfNecessityConverter;
import woolpack.typeconvert.SettingFnConverter;
import woolpack.typeconvert.ToArrayConverter;
import woolpack.typeconvert.ToCollectionDecompositionConverter;
import woolpack.typeconvert.ToCollectionViewConverter;
import woolpack.typeconvert.ToMapViewConverter;
import woolpack.typeconvert.ToTypeConverter;
import woolpack.utils.AppendableWriter;
import woolpack.utils.BeanUtils;
import woolpack.utils.DelegationCollection;
import woolpack.utils.DelegationMap;
import woolpack.utils.MapIterableMap;
import woolpack.validator.AddressedMessage;
import woolpack.validator.SimpleMessageCollector;
import woolpack.validator.ValidatorContext;

/**
 * コンストラクタとアクセサをテストします。
 * @author nakamura
 *
 */
public class AccessorTest extends TestCase {
	
	/**
	 * 以下の観点をテストします。
	 * コンストラクタすべて null でもよい。
	 * @param o
	 * @throws IllegalArgumentException
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	private void checkNullConsructors(final Object o) throws IllegalArgumentException, InstantiationException, IllegalAccessException, InvocationTargetException {
		final Class<?> clazz = o.getClass();
		final Constructor[] array = clazz.getConstructors();
		for(final Constructor c : array) {
			final Class[] paramTypes = c.getParameterTypes();
			final Object[] params = new Object[paramTypes.length];
			for(int i = 0; i < paramTypes.length; i++) {
				final Class paramType = paramTypes[i];
				if (paramType.isPrimitive()) {
					Object param = null;
					if (boolean.class.equals(paramType)) {
						param = false;
					} else if (char.class.equals(paramType)) {
						param = 'c';
					} else if (byte.class.equals(paramType)) {
						param = (byte)0;
					} else if (short.class.equals(paramType)) {
						param = (short)0;
					} else if (int.class.equals(paramType)) {
						param = (int)0;
					} else if (long.class.equals(paramType)) {
						param = (long)0;
					} else if (float.class.equals(paramType)) {
						param = (float)0;
					} else if (double.class.equals(paramType)) {
						param = (double)0;
					}
					params[i] = param;
				}
			}
			c.newInstance(params);
		}
	}
	
	/**
	 * 以下の観点をテストします。
	 * コンストラクタとゲッターメソッドの対応({@link BeanUtils#getConstructorGetterList}で試験される)。
	 * @param o
	 */
	public void checkGC(final Object o) {
		BeanUtils.getConstructorGetterList(o);
	}
	
	/**
	 * 以下の観点をテストします。
	 * コンストラクタとゲッターメソッドの対応({@link BeanUtils#getConstructorGetterList}で試験される)。
	 * ゲッターメソッドに対するセッターメソッドが存在する。
	 * @param o
	 * @throws IllegalArgumentException
	 * @throws SecurityException
	 * @throws InstantiationException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws NoSuchMethodException
	 */
	private void checkSC(final Object o) throws IllegalArgumentException, SecurityException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		final List<PropertyDescriptor> list = BeanUtils.getConstructorGetterList(o);
		final Object[] params = new Object[list.size()];
		final Class[] paramTypes = new Class[params.length];
		for (int i = 0; i < params.length; i++) {
			final PropertyDescriptor d = list.get(i);
			params[i] = d.getReadMethod().invoke(o, new Object[0]);
			paramTypes[i] = d.getPropertyType();
		}
		final Object o2 = o.getClass().getConstructor(paramTypes).newInstance(params);
		for (int i = 0; i < params.length; i++) {
			final PropertyDescriptor d = list.get(i);
			assertEquals(params[i], d.getReadMethod().invoke(o2, new Object[0]));
			d.getWriteMethod().invoke(o2, new Object[]{params[i]});
			assertEquals(params[i], d.getReadMethod().invoke(o2, new Object[0]));
		}
	}
	
	private void checkSCN(final Object o) throws IllegalArgumentException, InstantiationException, IllegalAccessException, InvocationTargetException, SecurityException, NoSuchMethodException {
		checkNullConsructors(o);
		checkSC(o);
	}
	
	public void testAccessor() throws Exception {
		// base
		checkSCN(FnUtils.delegate(FnUtils.fix(null)));
		
		// validator
		checkSCN(new AddressedMessage("a", 0, "b", "c"));
		{
			// getValueがあるので個別対応
			final ValidatorContext o = new ValidatorContext();
			final SimpleMessageCollector c = new SimpleMessageCollector();
			o.setCollectable(c);
			assertEquals(c, o.getCollectable());
		}
		
		// typeconvert
		checkSCN(new Converter(FnUtils.<Object, String>fix(null)));
		checkSCN(new DelegationIfNecessityConverter(FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new SettingFnConverter(FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new ToArrayConverter(FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new ToCollectionDecompositionConverter(FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new ToCollectionViewConverter(FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new ToMapViewConverter(FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new ToTypeConverter(FnUtils.<Class, Class>fix(null)));
		
		
		// el
		checkSC(new ArrayPathEL(new GettingEL[]{new PathEL("a")}, new PathEL("a")));
		checkSCN(new CachePropertyELFactory(ELUtils.PROPERTY_EL_FACTORY));
		checkSCN(new FixEL(new Object()));
		checkSCN(new MapEL("a", FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new MapPropertyELFactory(
				FnUtils.<ConvertContext, Void>fix(null),
				ELUtils.PROPERTY_EL_FACTORY));
		checkSC(new PathEL(
				"a",
				ELUtils.PROPERTY_EL_FACTORY,
				true));
		checkSCN(new PropertyEL(
				"a",
				ELUtils.PROPERTY_EL_FACTORY));
		checkSCN(new ReflectionPropertyELFactory(FnUtils.<ConvertContext, Void>fix(null)));
		checkSCN(new MapSimilarPropertyELFactory(
				FnUtils.<ConvertContext, Void>fix(null),
				ELUtils.PROPERTY_EL_FACTORY,
				FnUtils.<String, Iterable<String>>fix(null)));
		checkSCN(new SimilarPropertyELFactory(
				ELUtils.PROPERTY_EL_FACTORY,
				FnUtils.<String, Iterable<String>>fix(null)));
		checkSCN(new ThisEL(FnUtils.<ConvertContext, Void>fix(null), ""));
		checkSCN(new MaybeEL(new FixEL(new Object())));
		checkSCN(new MaybePropertyELFactory(
				true,
				ELUtils.PROPERTY_EL_FACTORY));
	}
	
	public void testUtils() throws IllegalArgumentException, SecurityException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		checkSCN(new AppendableWriter(new StringBuilder()));
		{
			// isEmptyがアクセサと誤解されるので個別対応
			final ArrayList<Object> list = new ArrayList<Object>();
			final DelegationCollection<Object> target = new DelegationCollection<Object>(list);
			assertEquals(list, target.getCollection());
			target.setCollection(list);
			assertEquals(list, target.getCollection());
		}
		{
			// isEmptyがアクセサと誤解されるので個別対応
			final HashMap<Object, Object> map = new HashMap<Object, Object>();
			final DelegationMap<Object, Object> target = new DelegationMap<Object, Object>(map);
			assertEquals(map, target.getMap());
			target.setMap(map);
			assertEquals(map, target.getMap());
		}
		{
			// isEmptyがアクセサと誤解されるので個別対応
			final ArrayList<Map<Object, Object>> list = new ArrayList<Map<Object, Object>>();
			final MapIterableMap<Object, Object> target = new MapIterableMap<Object, Object>(list);
			assertEquals(list, target.getIterable());
			target.setIterable(list);
			assertEquals(list, target.getIterable());
		}
	}
	
	public void testSql() throws IllegalArgumentException, SecurityException, InstantiationException, IllegalAccessException, InvocationTargetException, NoSuchMethodException {
		{
			// アクセサと誤解されるので個別対応
			final DataSource dataSource = new DataSourceAdapter(null);
			final TxBuilder builder = new TxBuilder(dataSource);
			assertEquals(dataSource, builder.getDataSource());
			builder.setDataSource(dataSource);
			assertEquals(dataSource, builder.getDataSource());
		}

		checkSCN(new PreparedStatementInfo(
				"",
				Arrays.asList("")));
	}
}
