/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.convert;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.Format;
import java.text.ParseException;
import java.text.ParsePosition;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 変換処理のユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 *
 */
public final class ConvertUtils {

	/**
	 * 入力を{@link Number}にキャストして
	 * {@link BigDecimal}に変換する関数です。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 */
	public static final Fn<Object, BigDecimal, RuntimeException> TO_BIG_DECIMAL = new Fn<Object, BigDecimal, RuntimeException>() {
		public BigDecimal exec(final Object c) {
			if (c instanceof BigDecimal) {
				return (BigDecimal) c;
			} else if (c instanceof BigInteger) {
				return new BigDecimal((BigInteger) c);
			} else {
				return new BigDecimal(((Number) c).doubleValue());
			}
		}
	};
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link BigInteger}に変換する関数です。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link ArithmeticException}を投げます。
	 */
	public static final Fn<Object, BigInteger, RuntimeException> TO_BIG_INTEGER = new Fn<Object, BigInteger, RuntimeException>() {
		public BigInteger exec(final Object c) {
			if (c instanceof BigInteger) {
				return (BigInteger) c;
			}
			if (c instanceof BigDecimal) {
				return ((BigDecimal) c).toBigIntegerExact();
			} else {
				return new BigDecimal(((Number) c).doubleValue()).toBigIntegerExact();
			}
		}
	};
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Byte}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 */
	public static final Fn<Object, Byte, RuntimeException> TO_BYTE = new Fn<Object, Byte, RuntimeException>() {
		public Byte exec(final Object c) {
			if (c instanceof Byte) {
				return (Byte) c;
			}
			final Number n = (Number) c;
			final byte b = n.byteValue();
			if (b != n.doubleValue()) {
				throw new IllegalArgumentException("out of range : " + c);
			}
			return b;
		}
	};
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Double}に変換する関数です。
	 * {@link Number#doubleValue()}を使用するため丸め誤差が発生する可能性があります。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 */
	public static final Fn<Object, Double, RuntimeException> TO_DOUBLE = new Fn<Object, Double, RuntimeException>() {
		public Double exec(final Object c) {
			if (c instanceof Double) {
				return (Double) c;
			}
			return ((Number) c).doubleValue();
		}
	};
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Float}に変換する関数です。
	 * 一度値を double 型で取得して float 型に変換するため、
	 * 丸め誤差が発生する可能性があります。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 */
	public static final Fn<Object, Float, RuntimeException> TO_FLOAT = new Fn<Object, Float, RuntimeException>() {
		public Float exec(final Object c) {
			if (c instanceof Float) {
				return (Float) c;
			}
			return ((Number) c).floatValue();
		}
	};
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Integer}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 */
	public static final Fn<Object, Integer, RuntimeException> TO_INTEGER = new Fn<Object, Integer, RuntimeException>() {
		public Integer exec(final Object c) {
			if (c instanceof Integer) {
				return (Integer) c;
			}
			final Number n = (Number) c;
			final int i = n.intValue();
			if (i != n.doubleValue()) {
				throw new IllegalArgumentException("out of range : " + c);
			}
			return i;
		}
	};
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Long}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 */
	public static final Fn<Object, Long, RuntimeException> TO_LONG = new Fn<Object, Long, RuntimeException>() {
		public Long exec(final Object c) {
			if (c instanceof Long) {
				return (Long) c;
			}
			final Number n = (Number) c;
			final long l = n.longValue();
			if (l != n.doubleValue()) {
				throw new IllegalArgumentException("out of range : " + c);
			}
			return l;
		}
	};
	
	/**
	 * 入力を{@link Number}にキャストして
	 * {@link Short}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 * 範囲外の場合は{@link IllegalArgumentException}を投げます。
	 */
	public static final Fn<Object, Short, RuntimeException> TO_SHORT = new Fn<Object, Short, RuntimeException>() {
		public Short exec(final Object c) {
			if (c instanceof Short) {
				return (Short) c;
			}
			final Number n = (Number) c;
			final short s = n.shortValue();
			if (s != n.doubleValue()) {
				throw new IllegalArgumentException("out of range : " + c);
			}
			return s;
		}
	};
	
	/**
	 * 入力を{@link java.util.Date}にキャストして
	 * {@link java.sql.Date}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 */
	public static final Fn<Object, java.sql.Date, RuntimeException> TO_SQL_DATE = new Fn<Object, java.sql.Date, RuntimeException>() {
		public java.sql.Date exec(final Object c) {
			if (c instanceof java.sql.Date) {
				return (java.sql.Date) c;
			} else {
				return new java.sql.Date(((java.util.Date) c).getTime());
			}
		}
	};
	
	/**
	 * 入力を{@link java.util.Date}にキャストして
	 * {@link Time}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 */
	public static final Fn<Object, Time, RuntimeException> TO_TIME = new Fn<Object, Time, RuntimeException>() {
		public Time exec(final Object c) {
			if (c instanceof Time) {
				return (Time) c;
			} else {
				return new Time(((Date) c).getTime());
			}
		}
	};
	
	/**
	 * 入力を{@link java.util.Date}にキャストして
	 * {@link Timestamp}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 */
	public static final Fn<Object, Timestamp, RuntimeException> TO_TIMESTAMP = new Fn<Object, Timestamp, RuntimeException>() {
		public Timestamp exec(final Object c) {
			if (c instanceof Timestamp) {
				return (Timestamp) c;
			} else {
				return new Timestamp(((Date) c).getTime());
			}
		}
	};
	
	/**
	 * 入力を{@link Date}にキャストして
	 * {@link Calendar}に変換する関数です。
	 * キャストできない場合は{@link ClassCastException}を投げます。
	 */
	public static final Fn<Object, Calendar, RuntimeException> TO_CALENDAR = new Fn<Object, Calendar, RuntimeException>() {
		public Calendar exec(final Object c) {
			if (c instanceof Calendar) {
				return (Calendar) c;
			} else {
				final Calendar calendar = Calendar.getInstance();
				calendar.setTime((Date) c);
				return calendar;
			}
		}
	};
	
	/**
	 * {@link Object#toString()}で変換する関数です。
	 */
	public static final Fn<Object, String, RuntimeException> TO_STRING = new Fn<Object, String, RuntimeException>() {
		public String exec(final Object c) {
			return c == null ? null : c.toString();
		}
	};
	
	private ConvertUtils() {
	}
	
	/**
	 * {@link Map#keySet()}を返す関数を生成します。
	 * @param <K>
	 * @return 関数。
	 */
	public static <K> Fn<Map<K, ?>, Set<K>, RuntimeException> keySet() {
		return new Fn<Map<K, ?>, Set<K>, RuntimeException>() {
			public Set<K> exec(final Map<K, ?> c) {
				return c.keySet();
			}
		};
	}
	
	/**
	 * {@link Format}の新規のインスタンスを{@link Format#clone()}により複製するファクトリを生成します。
	 * {@link java.text.Format}の各サブクラスはスレッドセーフでなく
	 * {@link java.lang.Cloneable}を実装して Prototype デザインパターンを適用していますが、
	 * 本クラスはこれを Abstract Factory デザインパターンに変換しています。
	 * <br/>適用しているデザインパターン：{@link Format}のAbstract Factory、Prototype。
	 * @param format プロトタイプ。
	 * @return 関数。
	 */
	public static Fn<Object, Format, RuntimeException> formatFactory(final Format format) {
		return new Fn<Object, Format, RuntimeException>() {
			public Format exec(final Object c) {
				return (Format) format.clone();
			}
		};
	}
	
	/**
	 * {@link Format#format(Object)}で変換する関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param <E>
	 * @param formatFactory フォーマットのファクトリ。
	 * @return 関数。
	 */
	public static <E extends Exception> Fn<Object, String, E> format(
			final Fn<?, ? extends Format, ? extends E> formatFactory) {
		return new Fn<Object, String, E>() {
			public String exec(final Object c) throws E {
				return formatFactory.exec(null).format(c);
			}
		};
	}
	
	/**
	 * コンテキスト役をIPv4アドレスとしてネットワークアドレスを返す関数を生成します。
	 * <br/>適用しているデザインパターン：ネットマスクとアドレスのCurrying。
	 * @param netMask ネットマスク。
	 * @return 関数。
	 */
	public static Fn<String, String, RuntimeException> netMask(final String netMask) {
		return new Fn<String, String, RuntimeException>() {
			private final int IP_COUNT = 4;
			private final Pattern PATTERN = Pattern.compile("(\\d+)\\.(\\d+)\\.(\\d+)\\.(\\d+)");
			private int[] maskArray = toArray(netMask);
			private int[] toArray(final String s) {
				if (s == null) {
					return null;
				}
				final Matcher m = PATTERN.matcher(s);
				if (!m.matches()) {
					return null;
				}
				final int[] a = new int[IP_COUNT];
				for (int i = 0; i < a.length; i++) {
					a[i] = Integer.parseInt(m.group(i + 1));
				}
				return a;
			}
			public String exec(final String c) {
				final int[] a = toArray(c);
				if (a == null) {
					return null;
				}

				for (int i = 0; i < a.length; i++) {
					a[i] = a[i] & maskArray[i];
				}

				final StringBuilder sb = new StringBuilder();
				for (int i = 0; i < a.length; i++) {
					if (i != 0) {
						sb.append('.');
					}
					sb.append(a[i]);
				}

				return sb.toString();
			}
		};
	}
	
	/**
	 * {@link Format#parseObject(String)}で変換する関数を生成します。
	 * 「NumberFormat の構文解析問題の解決」
	 * (http://www-06.ibm.com/jp/developerworks/java/library/j-numberformat/index.shtml)
	 * のテクニックを使用してデータの欠損を検出しています。
	 * パースに失敗した場合は{@link java.text.ParseException}を投げます。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param formatFactory フォーマットのファクトリ。
	 * @return 関数。
	 */
	public static Fn<String, Object, Exception> parse(final Fn<?, ? extends Format, ? extends Exception> formatFactory) {
		return new Fn<String, Object, Exception>() {
			public Object exec(final String c) throws Exception {
				final ParsePosition pp = new ParsePosition(0);
				final Format format = formatFactory.exec(null);
				final Object o = format.parseObject(c, pp);
				if (o != null && pp.getIndex() == c.length()) {
					return o;
				} else {
					throw new ParseException("cannot parse " + c + " at " + format, 0);
				}
			}
		};
	}
	
	/**
	 * 正規表現で変換する関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param pattern 正規表現。
	 * @param replacement 変換パターン。
	 * @return 関数。
	 */
	public static Fn<String, String, RuntimeException> convertRegExp(
			final Pattern pattern,
			final String replacement) {
		return new Fn<String, String, RuntimeException>() {
			public String exec(final String c) {
				return pattern.matcher(c).replaceAll(replacement);
			}
		};
	}
	
	/**
	 * 成功するまで順に委譲する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param iterable 委譲先の一覧。
	 * @param errorFn エラー発生時の委譲先。
	 * @param finalFn 最後まで成功しなかった場合の委譲先。
	 * @return 関数。
	 */
	public static <C, R, E extends Exception> Fn<C, R, E> trys(
			final Iterable<? extends Fn<? super C, ? extends R, ? extends Exception>> iterable,
			final Fn<? super Exception, ?, ? extends E> errorFn,
			final Fn<? super Exception, ? extends R, ? extends E> finalFn) {
		return new Fn<C, R, E>() {
			public R exec(final C c) throws E {
				Exception t = null;
				for (final Fn<? super C, ? extends R, ? extends Exception> fn : iterable) {
					try {
						return fn.exec(c);
					} catch (final Exception e) {
						t = e;
						errorFn.exec(t);
					}
				}
				return finalFn.exec(t);
			}
		};
	}
	
	/**
	 * 成功するまで順に委譲する関数を生成します。
	 * 最後まで成功しなかった場合は{@link IllegalStateException}を投げます。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param iterable 委譲先の一覧。
	 * @return 関数。
	 * @see #trys(Iterable, Fn, Fn)
	 */
	public static <C, R> Fn<C, R, IllegalStateException> trys(
			final Iterable<? extends Fn<? super C, ? extends R, ? extends Exception>> iterable) {
		return trys(
				iterable,
				FnUtils.<Exception, R, IllegalStateException>fixThrows(null),
				new Fn<Exception, R, IllegalStateException>() {
					public R exec(final Exception c) throws IllegalStateException {
						throw new IllegalStateException(c);
					}
				});
	}
	
	/**
	 * マップの指定されたキーを残す関数を生成します。
	 * <br/>適用しているデザインパターン：変換ルールと変換対象のCurrying。
	 * @param collection 残すキーのコレクション。
	 * @return 関数。
	 */
	public static Fn<Map<?, ?>, Boolean, RuntimeException> retainKeys(final Collection<?> collection) {
		return new Fn<Map<?, ?>, Boolean, RuntimeException>() {
			public Boolean exec(final Map<?, ?> c) {
				c.keySet().retainAll(collection);
				return true;
			}
		};
	}
}
