/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.el;

import java.util.Arrays;
import java.util.Collection;

/**
 * 順に基点をたどっていく式言語です。
 * <br/>適用しているデザインパターン：{@link GettingEL}のComposite。
 * 
 * @author nakamura
 * 
 */
public class ArrayPathEL extends AbstractEL {
	private GettingEL[] getting;
	private EL setting;

	/**
	 * @param getting 基点をたどっていくための式言語の一覧。
	 * @param setting 値を設定または取得するための式言語。
	 */
	public ArrayPathEL(final GettingEL[] getting, final EL setting) {
		super();
		this.getting = getting;
		this.setting = setting;
	}

	/**
	 * @param getting 基点をたどっていくための式言語。
	 * @param setting 値を設定または取得するための式言語。
	 */
	public ArrayPathEL(final GettingEL getting, final EL setting) {
		this(new GettingEL[] { getting }, setting);
	}

	/**
	 * @param getting 基点をたどっていくための式言語の一覧。
	 * @param setting 値を設定または取得するための式言語。
	 */
	public ArrayPathEL(
			final Collection<? extends GettingEL> getting,
			final EL setting) {
		this(getting.toArray(new GettingEL[0]), setting);
	}

	/**
	 * @param setting 値を設定または取得するための式言語の一覧。末尾の式言語は値を設定または取得するために使用します。
	 */
	public ArrayPathEL(final EL... setting) {
		this(Arrays.asList(setting).subList(0, setting.length - 1),
				setting[setting.length - 1]);
	}

	/**
	 * @param setting 値を設定または取得するための式言語の一覧。末尾の式言語は値を設定または取得するために使用します。
	 */
	public ArrayPathEL(final Collection<? extends EL> setting) {
		this(setting.toArray(new EL[0]));
	}

	private Object getLocal(final Object root) {
		Object r = root;
		for (int i = 0; i < getting.length; i++) {
			r = getting[i].getValue(r);
		}
		return r;
	}

	@Override
	public boolean setValue(final Object root, final Object value) {
		return setting.setValue(getLocal(root), value);
	}

	@Override
	public Object getValue(final Object root, final Class toType) {
		return setting.getValue(getLocal(root), toType);
	}

	public GettingEL[] getGetting() {
		return getting;
	}
	public void setGetting(final GettingEL[] getting) {
		this.getting = getting;
	}
	public EL getSetting() {
		return setting;
	}
	public void setSetting(final EL setting) {
		this.setting = setting;
	}
}
