/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.misc;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Semaphore;

import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * 多雑な部品のユーティリティです。
 * 
 * @author nakamura
 *
 */
public final class MiscUtils {
	private MiscUtils() {
	}
	
	/**
	 * 委譲先の実行時間を測定する関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param fn 委譲先。
	 * @param reportFn 結果の送付先。
	 * @return 関数。
	 */
	public static <C, R, E extends Exception> Fn<C, R, E> lapTime(
			final Fn<? super C, ? extends R, ? extends E> fn,
			final Fn<? super Long, ?, ? extends E> reportFn) {
		return new Fn<C, R, E>() {
			public R exec(final C c) throws E {
				final long s = System.currentTimeMillis();
				try {
					return fn.exec(c);
				} finally {
					reportFn.exec(System.currentTimeMillis() - s);
				}
			}
		};
	}
	
	/**
	 * 複数のスレッドで同時に実行することができない{@link Fn}を、
	 * 複数生成することにより並行に実行するようにみせる関数を生成します。
	 * <br/>適用しているデザインパターン：内部で{@link Fn}のCompositeを生成して使用するBuilder。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param <E1>
	 * @param factory 複数のスレッドで同時に実行できない{@link Fn}のファクトリ。
	 * @param length 並行実行数。
	 * @return 関数。
	 * @throws IllegalArgumentException length がゼロ以下の場合。
	 * @throws Exception {@link Fn}の生成に失敗した場合。
	 */
	public static <C, R, E extends Exception, E1 extends Exception> Fn<C, R, E> loadBalancer(
			final Fn<Object, ? extends Fn<? super C, ? extends R, ? extends E>, ? extends E1> factory,
			final int length) throws Exception {
		if (length < 1) {
			throw new IllegalArgumentException("length must be equals or more than 1 but " + length);
		}
		final Semaphore[] semaphoreArray = new Semaphore[length];
		final List<Fn<? super C, ? extends R, ? extends E>> fnList = new ArrayList<Fn<? super C, ? extends R, ? extends E>>(length);
		for (int i = 0; i < length; i++) {
			semaphoreArray[i] = new Semaphore(1);
			synchronized (factory) {
				fnList.add(factory.exec(null));
			}
		}
		return new Fn<C, R, E>() {
			// position の原子性を厳密に制御する必要はない。
			private int position = 0;
			public R exec(final C c) throws E {
				int i = position;
				while (true) {
					for (; i < length; i++) {
						if (semaphoreArray[i].tryAcquire()) {
							try {
								return fnList.get(i).exec(c);
							} finally {
								semaphoreArray[i].release();
								position = (i + 1) % length;
							}
						}
						Thread.yield();
					}
					i = 0;
				}
			}
		};
	}
	
	/**
	 * {@link Fn}に委譲する{@link Runnable}を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のAdapter。
	 * @param <C>
	 * @param fn 委譲先。
	 * @param c 委譲先に渡す値。
	 * @return ランナブル。
	 */
	public static <C> Runnable fnRunnable(final Fn<? super C, ?, ? extends RuntimeException> fn, final C c) {
		return new Runnable() {
			public void run() {
				fn.exec(c);
			}
		};
	}
	
	/**
	 * {@link Runnable}に委譲する関数を生成します。
	 * {@link Fn#exec(Object)}は null を返します。
	 * <br/>適用しているデザインパターン：{@link Runnable}のAdapter。
	 * @param <R>
	 * @param runnable ランナブル。
	 * @return 関数。
	 */
	public static <R> Fn<Object, R, RuntimeException> runnableFn(final Runnable runnable) {
		return new Fn<Object, R, RuntimeException>() {
			public R exec(final Object c) {
				runnable.run();
				return null;
			}
		};
	}
	
	/**
	 * スリープする関数を生成します。
	 * @param <C>
	 * @param <R>
	 * @param sleepMillis スリープ時間。
	 * @return 関数。
	 */
	public static <C, R> Fn<C, R, InterruptedException> sleep(final long sleepMillis) {
		return new Fn<C, R, InterruptedException>() {
			public R exec(final C c) throws InterruptedException {
				if (sleepMillis > 0) {
					Thread.sleep(sleepMillis);
				}
				return null;
			}
		};
	}
	
	/**
	 * 一覧から指定されたロケールに最も近いロケールをキーとして値を返す関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のProxy。
	 * @param <R>
	 * @param <E>
	 * @param map マップ。
	 * @param defaultFn 引数に対応する値が null の場合の委譲先。
	 * @return 関数。
	 */
	public static <R, E extends Exception> Fn<Locale, R, E> switchNearLocale(
			final Map<? super Locale, ? extends R> map,
			final Fn<? super Locale, ? extends R, ? extends E> defaultFn) {
		return new Fn<Locale, R, E>() {
			private static final int LOCALE_DEPTH = 3;
			private int calcLocaleCount(final Locale key) {
				return (key.getVariant() != null && key.getVariant().length() > 0)
				? LOCALE_DEPTH
						: (key.getCountry() != null
								&& key.getCountry().length() > 0) ? 2
								: 1;
			}
			public R exec(final Locale c) throws E {
				{
					final R r = map.get(c);
					if (r != null) {
						return r;
					}
				}
				if (c == null) {
					return defaultFn.exec(c);
				}
				
				final int keyCount = calcLocaleCount(c);

				Locale l0 = null;
				int count0 = 0;
				for (final Object object : map.keySet()) {
					if (!(object instanceof Locale)) {
						continue;
					}
					final Locale l1 = (Locale) object;
					if (keyCount < calcLocaleCount(l1)) {
						continue;
					}
					int count1 = 0;
					if (c.getLanguage() != null
							&& c.getLanguage().equals(l1.getLanguage())) {
						count1++;
						if (c.getCountry() != null
								&& c.getCountry().equals(l1.getCountry())) {
							count1++;
						}
					}
					if (count0 < count1) {
						count0 = count1;
						l0 = l1;
					}
				}
				if (l0 == null) {
					return defaultFn.exec(c);
				}
				return map.get(l0);
			}
		};
	}
	
	/**
	 * {@link ThreadLocal#get()}を返す関数を生成します。
	 * <br/>適用しているデザインパターン：{@link ThreadLocal}のAdapter。
	 * @param <R>
	 * @param threadLocal スレッドローカル。
	 * @return 関数。
	 */
	public static <R> Fn<Object, R, RuntimeException> threadLocal(final ThreadLocal<? extends R> threadLocal) {
		return new Fn<Object, R, RuntimeException>() {
			public R exec(final Object c) {
				return threadLocal.get();
			}
		};
	}
	
	/**
	 * {@link Locale}を元に id に接尾辞を付加して成功するまで委譲する関数を生成します。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite、{@link Fn}のProxy。
	 * @param <R>
	 * @param <E>
	 * @param fn 委譲先。　
	 * @param threadLocal 現在のスレッドの{@link Locale}を保持するスレッドローカル。
	 * @param errorFn エラー発生時の委譲先。
	 * @param finalFn 全て成功しなかった場合の委譲先。
	 * @return 関数。
	 */
	public static <R, E extends Exception> Fn<String, R, E> tryLocales(
			final Fn<? super String, ? extends R, ? extends Exception> fn,
			final ThreadLocal<Locale> threadLocal,
			final Fn<? super Exception, ?, ? extends E> errorFn,
			final Fn<? super Exception, ? extends R, ? extends E> finalFn) {
		return new Fn<String, R, E>() {
			private String removeRight(final String s, final int i) {
				return s.substring(0, s.length() - i);
			}
			public R exec(final String id) throws E {
				final Locale l = threadLocal.get();
				final Locale d = Locale.getDefault();

				final String[] idArray;
				if (l == null) {
					idArray = new String[4];
					idArray[0] = id + '_' + d.getLanguage() + '_' + d.getCountry() + '_' + d.getVariant();
					idArray[1] = removeRight(idArray[0], d.getVariant().length() + 1);
					idArray[2] = removeRight(idArray[1], d.getCountry().length() + 1);
					idArray[3] = id;
				} else {
					idArray = new String[7];
					idArray[0] = id + '_' + l.getLanguage() + '_' + l.getCountry() + '_' + l.getVariant();
					idArray[1] = removeRight(idArray[0], l.getVariant().length() + 1);
					idArray[2] = removeRight(idArray[1], l.getCountry().length() + 1);
					idArray[3] = id + '_' + d.getLanguage() + '_' + d.getCountry() + '_' + d.getVariant();
					idArray[4] = removeRight(idArray[3], d.getVariant().length() + 1);
					idArray[5] = removeRight(idArray[4], d.getCountry().length() + 1);
					idArray[6] = id;
				}
				Exception e0 = null;
				for (int i = 0; i < idArray.length; i++) {
					try {
						final R r = fn.exec(idArray[i]);
						if (r != null) {
							return r;
						}
					} catch (final Exception e) {
						e0 = e;
						errorFn.exec(e0);
					}
				}
				return finalFn.exec(e0);
			}
		};
	}

	/**
	 * {@link Locale}を元に id に接尾辞を付加して成功するまで委譲する関数を生成します。
	 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
	 * 全て成功しなかった場合は{@link IllegalStateException}を投げます。
	 * @param <R>
	 * @param fn 委譲先。　
	 * @param threadLocal 現在のスレッドの{@link Locale}を保持するスレッドローカル。
	 * @return 関数。
	 * @see #tryLocales(Fn, ThreadLocal, Fn, Fn)
	 */
	public static <R> Fn<String, R, IllegalStateException> tryLocales(
			final Fn<? super String, ? extends R, ? extends Exception> fn,
			final ThreadLocal<Locale> threadLocal) {
		return MiscUtils.<R, IllegalStateException>tryLocales(
				fn,
				threadLocal,
				FnUtils.<Exception, R, IllegalStateException>fixThrows(null),
				new Fn<Exception, R, IllegalStateException>() {
					public R exec(final Exception c) throws IllegalStateException {
						throw new IllegalStateException(c);
					}
				});
	}
	
	/**
	 * 成功するまでリトライする関数を生成します。
	 * <br/>適用しているデザインパターン：{@link Fn}のComposite。
	 * @param <C>
	 * @param <R>
	 * @param <E>
	 * @param fn 委譲先。
	 * @param count リトライ回数。
	 * @param errorFn エラー発生時の委譲先。
	 * @param finalFn 全て成功しなかった場合の委譲先。
	 * @return 関数。
	 */
	public static <C, R, E extends Exception> Fn<C, R, E> retry(
			final Fn<? super C, ? extends R, ? extends Exception> fn,
			final int count,
			final Fn<? super Exception, ?, ? extends E> errorFn,
			final Fn<? super Exception, ? extends R, ? extends E> finalFn) {
		return new Fn<C, R, E>() {
			public R exec(final C c) throws E {
				Exception t = null;
				for (int i = 0; i < count; i++) {
					try {
						return fn.exec(c);
					} catch (final Exception e) {
						t = e;
						errorFn.exec(t);
					}
				}
				return finalFn.exec(t);
			}
		};
	}
	
	/**
	 * 成功するまでリトライする関数を生成します。
	 * 全て成功しなかった場合は{@link IllegalStateException}を投げます。
	 * @param <C>
	 * @param <R>
	 * @param fn 委譲先。
	 * @param count リトライ回数。
	 * @return 関数。
	 * @see #retry(Fn, int, Fn, Fn)
	 */
	public static <C, R> Fn<C, R, IllegalStateException> retry(
			final Fn<? super C, ? extends R, ? extends Exception> fn,
			final int count) {
		return retry(
				fn,
				count,
				FnUtils.<Exception, R, IllegalStateException>fixThrows(null),
				new Fn<Exception, R, IllegalStateException>() {
					public R exec(final Exception c) throws IllegalStateException {
						throw new IllegalStateException(c);
					}
				});
	}
}
