/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.validator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import junit.framework.TestCase;
import woolpack.bool.BoolUtils;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.utils.Utils;

public class NameBranchIfExistsTest extends TestCase {
	private List<Object> contextList;
	private Fn<ValidatorContext, Boolean, RuntimeException> fn;
	
	@Override
	public void setUp() throws Exception {
		super.setUp();
		contextList = new ArrayList<Object>();
		fn = ValidatorUtils.branchByNameIfExists(
				BoolUtils.AND,
				Utils
				.map("a", FnUtils.join(ValidatorUtils.VALUE, FnUtils.recode(FnUtils.fix(false), contextList, null)))
				.map("b", FnUtils.join(ValidatorUtils.VALUE, FnUtils.recode(FnUtils.fix(true), contextList, null))));
	}

	public void testException() {
		final Fn<ValidatorContext, Boolean, RuntimeException> fn = ValidatorUtils.branchByNameIfExists(
				BoolUtils.AND,
				Utils.map("a", FnUtils.<ValidatorContext, Boolean, RuntimeException>throwing(new IllegalStateException())));
		final ValidatorContext context = new ValidatorContext();
		context.setKey("h");
		context.setInputMap(Utils.map("a", Arrays.asList((Object) "va")));
		try {
			fn.exec(context);
			fail();
		} catch (final IllegalStateException e) {
		}
		assertEquals("h", context.getKey());
	}

	public void testNotMatch() {
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils
				.map("x", Arrays.asList((Object) "vx"))
				.map("y", Arrays.asList((Object) "vy")));
		assertTrue(fn.exec(context));
		assertEquals(0, contextList.size());
	}

	public void testMatchFalse() {
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils
				.map("a", Arrays.asList((Object) "va"))
				.map("y", Arrays.asList((Object) "vy")));
		assertFalse(fn.exec(context));
		assertEquals(Arrays.asList("va"), contextList);
	}

	public void testMatchTrue() {
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils
				.map("x", Arrays.asList((Object) "vx"))
				.map("b", Arrays.asList((Object) "vb")));
		assertTrue(fn.exec(context));
		assertEquals(Arrays.asList("vb"), contextList);
	}

	public void testMatchFalse2() {
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils
				.map("a", Arrays.asList((Object) "va"))
				.map("b", Arrays.asList((Object) "vb")));
		assertFalse(fn.exec(context));
		assertEquals(Arrays.asList("va", "vb"), contextList);
	}

	public void testMatchFalseEmpty() {
		final ValidatorContext context = new ValidatorContext();
		context.setInputMap(Utils
				.map("a", Arrays.asList((Object) ""))
				.map("b", Arrays.asList((Object) "")));
		assertFalse(fn.exec(context));
		assertEquals(Arrays.asList("", ""), contextList);
	}
}
