<?php 
/* 
Plugin Name: Ktai Location
Plugin URI: http://www.yuriko.net/travel/2007/01/18/1785/
Description: Read a latitude-longtitude data from a map URL which is provided with location services for mobile phone, or from an EXIF infomation of a posted picture. And save the location data as a "Lat_Long" custom field.
Author: IKEDA yuriko
Version: 0.7.1
Author URI: http://www.yuriko.net/travel/category/wordpress/
*/

/*  Copyright (c) 2007 yuriko

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Special Thanks:
	- "TSJ Blog II": author of wp-eznavi.
		[http://blog2.atsj.net/]
*/

define ('CSS_CLASS_FOR_LOCATION', 'locationurl');
define ('GEO_META_FIELD_NAME', 'Lat_Long');
define ('EZNAVI_WGS84',  0);
define ('EZNAVI_TOKYO',  1);
define ('EZNAVI_ITRF',   2);
define ('EZNAVI_DMS',    0);
define ('EZNAVI_DEGREE', 1);

add_action('wp_head',      array('Ktai_Location', 'output_style'), 20);
//add_action('save_post',    array('Ktai_Location', 'read_location'), 20);
add_action('publish_post', array('Ktai_Location', 'read_location'), 20);

/* ==================================================
 *   Ktai_Location Class
   ================================================== */

class Ktai_Location {
	public $lat;
	public $lon;
	public $url;
	public $datum;
	public $unit;
	public $accuracy;

/* ==================================================
 * Add custom style sheet into XHTML header.
 * @param	none
 * @return	none
*/
static public function output_style() {
?>
<style type="text/css">
<?php echo '.' . CSS_CLASS_FOR_LOCATION . "{\n"; ?>
	display:none;
}
</style>
<?php
}

/* ==================================================
 * Read location info from post content.
 * @param	Int $postID
 * @return	Int $postID
*/
static public function read_location($postID) {
	if (! is_numeric($postID)) {
		return $postID;
	}

	$post = get_post($postID);
	$location = self::factory($post->post_content);
	if (! $location) {
		return $postID;
	}
	if ($location->lat && $location->lon) {
		$do_update = $location->update_meta($postID);
		if ($do_update && $location->url) {
			$location->format_url($post);
		}
	}
	return  $postID;
}

/* ==================================================
 * Insert/Update the location info of the postmeta table of the DB.
 * @param	Int     $postID
 * @return	Boolean $do_update
*/
private function update_meta($postID) {
	$do_update = FALSE;
	$meta_value = get_post_meta($postID, GEO_META_FIELD_NAME, TRUE);
	if ($meta_value) {
		$latlng = explode(',', $meta_value);
		if (! intval($latlng[0]) && ! intval($latlng[1])) {
			update_post_meta($postID, GEO_META_FIELD_NAME, $this->lat . ',' . $this->lon);
			$do_update = TRUE;
		}
	} else {
		add_post_meta($postID, GEO_META_FIELD_NAME, $this->lat . ',' . $this->lon);
		$do_update = TRUE;
	}
	return $do_update;
}

/* ==================================================
 * Add div element into the location URL of the post content.
 * @param	Object $post
 * @return	none
*/
protected function format_url ($post) {
	$new_url = '<div class="' . CSS_CLASS_FOR_LOCATION . '">' . str_replace('http://', 'HTTP://', $this->url) . '</div>';
	$new_content = str_replace($this->url, $new_url, $post->post_content, $count);
	if ($count) {
		global $wpdb;
		$wpdb->query("UPDATE $wpdb->posts SET post_content = '" . $wpdb->escape($new_content) . "' WHERE ID = " . $post->ID);
	}
	return;
}

/* ==================================================
 * Parse an HTTP GET parameters in the request URL.
 * @param	String $query
 * @return	Array $params
*/
protected function get_params($query) {
	$query = str_replace('&amp;', '&', $query);
	parse_str(urldecode($query), $params);
	return $params;
}

/* ==================================================
 * Convert a degree-minites-second format into degree format.
 * @param	String $dms
 * @param	Int    $limit
 * @return	Float  $value
*/
protected function dms2deg($dms, $limit) {
	$value = NULL;
	if (preg_match("#([-+]?)(\d+)\.(\d+)\.(\d+)\.(\d+)#", $dms, $m)) {
		$value = ($m[1] == '-' ? -1 : 1) *
			 (intval($m[2]) + 
			  intval($m[3]) / 60 + 
			  intval($m[4]) / 3600 + 
			  intval($m[5]) / 360000);
	} elseif (is_numeric($str)) {
		$value = floatval($str);
	}
	if ($value < -1 * $limit || $value > $limit) {
		$value = NULL;
	}
	return $value;
}

/* ==================================================
 * Convert a location info from Tokyo geometry to WGS84 geometry.
 * @param	none
 * @return	none
*/
protected function tokyo2wgs84() {
	/*
	 * Thanks to "Mac･GPS･Perl"
	 * <http://homepage3.nifty.com/Nowral/02_DATUM/02_DATUM.html>
	 */
	$this->lat = $this->lat - 0.000106950 * $this->lat + 0.000017464 * $this->lon + 0.0046017;
	$this->lon = $this->lon - 0.000046038 * $this->lat - 0.000083043 * $this->lon + 0.0100400;
	return;
}

/* ==================================================
 * Validate the location value.
 * @param	none
 * @return	none
*/
protected function fix_value() {
	if (is_numeric($this->lat)) {
		$this->lat = floatval($this->lat);
	} else {
		$this->lat = NULL;
	}
	if (is_numeric($this->lon)) {
		$this->lon = floatval($this->lon);
	} else {
		$this->lon = NULL;
	}
	return;
}

/* ==================================================
 * Create a location info object along location services.
 * @param	String $content
 * @return	Object $loc
*/
static public function factory($content) {
	if (preg_match('#^http://#m', $content)) {
		$loc = Ktai_Location_EZ_Navi::factory($content);
		if (! is_null($loc)) {
			$loc->fix_value();;
			return $loc;
		}
		$loc = Ktai_Location_iMapFan::factory($content);
		if (! is_null($loc)) {
			$loc->fix_value();;
			return $loc;
		}
		$loc = Ktai_Location_iMapion::factory($content);
		if (! is_null($loc)) {
			$loc->fix_value();;
			return $loc;
		}
		$loc = Ktai_Location_iZenrin::factory($content);
		if (! is_null($loc)) {
			$loc->fix_value();;
			return $loc;
		}
	}
	if (preg_match('/<img /', $content)) {
		$loc = Ktai_Location_EXIF::factory($content);
		if (! is_null($loc)) {
			$loc->fix_value();;
			return $loc;
		}
	}
	return NULL;
}

// ==================================================
} // End of class

/* ==================================================
 *   Ktai_Location_EXIF Class
   ================================================== */

class Ktai_Location_EXIF extends Ktai_Location {

static public function factory($content) {
	if (! preg_match_all('/(<a [^>]*href="(.*?)"[^>]*>)?<img [^>]*src="(.*?)"/', $content, $matches, PREG_SET_ORDER)
	||  ! function_exists('exif_read_data')) {
		return NULL;
	}
	$loc = NULL;
	foreach ($matches as $m) {
		if ($m[1]) {
			$file = self::decide_file_path($m[2]);
		} else {
			$file = self::decide_file_path($m[3]);
		}
		if (! $file) {
			continue;
		}
		$exif = exif_read_data($file, 0, TRUE);
		if ($exif && array_key_exists('GPS', $exif)) {
			$loc = new Ktai_Location_EXIF;
			$loc->lat   = $loc->decode_exif_location($exif['GPS']['GPSLatitude'], $exif['GPS']['GPSLatitudeRef']);
			$loc->lon   = $loc->decode_exif_location($exif['GPS']['GPSLongitude'], $exif['GPS']['GPSLongitudeRef']);
			$loc->datum = $exif['GPS']['GPSMapDatum'];
			if ($loc->lat && $loc->lon && $loc->datum == 'TOKYO') {
				$loc->tokyo2wgs84();
			}
            break;
		}
	}
	return $loc;
}

/* ==================================================
 * Decide the filepath from html img src attributes.
 * @param	String $src
 * @return	String $path
*/

static private function decide_file_path($src) {
	$blog_url = get_bloginfo('url');
	foreach (array(get_bloginfo('url'), get_settings('siteurl')) as $url) {
		if (! $url) {
			continue;
		}
		$src = preg_replace('#^' . quotemeta($url) . '#', '', $src, 1, $count);
		if ($count && file_exists($path = ABSPATH . $src)) {
			return $path;
		}
		$short_url = preg_replace('#^https?://[^/]*/#', '/', $url);
		$src = preg_replace('#^' . quotemeta($short_url) . '#', '', $src, 1, $count);
		if ($count && file_exists($path = ABSPATH . $src)) {
			return $path;
		}
	}
	$src = preg_replace('#^https?://[^/]*/#', '', $src, 1, $count);
	if ($count && file_exists($path = ABSPATH . $src)) {
		return $path;
	}
	if (file_exists($path = ABSPATH . $src)) {
		return $path;
	}
	return NULL;
}

/* ==================================================
 * Decode a location value of EXIF format.
 * @param	Array  $dms
 * @param   String $ref
 * @return	String $degree
*/

static private function decode_exif_location($dms, $ref) {
	if (count($dms) != 3) {
		return NULL;
	}
    $deg = array_map('intval', explode('/', $dms[0]));
    $min = array_map('intval', explode('/', $dms[1]));
    $sec = array_map('intval', explode('/', $dms[2]));
	return ($ref == 'S' || $ref == 'W' ? -1 : 1) *
	  ( ($deg[1] != 0 ? $deg[0] / $deg[1] : 0)
	  + ($min[1] != 0 ? $min[0] / ($min[1] * 60) : 0)
	  + ($sec[1] != 0 ? $sec[0] / ($sec[1] * 3600) : 0));
}

} // End of class

/* ==================================================
 *   Ktai_Location_EZ_Navi Class
 * example: http://walk.eznavi.jp/map/?datum=AAA&unit=BBB&lat=XXX&lon=YYY
   ================================================== */

class Ktai_Location_EZ_Navi extends Ktai_Location {

static public function factory($content) {
	if (! preg_match('#(\[GPS.*?URL\])?\s*http://walk\.eznavi\.jp/map/\?(\S+)#s', $content, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_EZ_Navi;
	$loc->url = $m[0];
	$params = parent::get_params($m[2]);
	$loc->datum    = @$params['datum'];
	$loc->unit     = @$params['unit'];
	$loc->lat      = trim(@$params['lat']);
	$loc->lon      = trim(@$params['lon']);
	$loc->accuracy = @$params['fm'];
	if ($loc->unit == EZNAVI_DMS) {
		$loc->lat = parent::dms2deg($loc->lat, 90);
		$loc->lon = parent::dms2deg($loc->lon, 180);
	}
	if (isset($loc->datum) && $loc->datum == EZNAVI_TOKYO) {
		$loc->tokyo2wgs84();
	}
	return $loc;
}

} // End of class

/* ==================================================
 *   Ktai_Location_iMapFan Class
 * example: http://i.mapfan.com/m.cgi?uid=NULLGWDOCOMO&F=AP&M=E139.34.4.3N35.36.50.6&SC=SY3JY8GE&AR=07900
   ================================================== */

class Ktai_Location_iMapFan extends Ktai_Location {

static public function factory($content) {
	if (! preg_match('#http://i\.mapfan\.com/m\.cgi\?uid=NULLGWDOCOMO&(\S+)#m', $content, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_iMapion;
	$loc->url = $m[0];
	$params = parent::get_params($m[1]);
	preg_match('/^([EW])([.0-9]+)([NS])([.0-9]+)$/', @$params['M'], $m);
	$loc->lon = ($m[1] == 'W' ? -1: 1) * parent::dms2deg($m[2], 180);
	$loc->lat = ($m[3] == 'S' ? -1: 1) * parent::dms2deg($m[4], 90);
	$loc->tokyo2wgs84();
	return $loc;
}

} // End of class

/* ==================================================
 *   Ktai_Location_iMapion Class
 * example: http://i.mapion.co.jp/c/f?uc=1&nl=35/36/02.038&el=139/30/38.558&grp=mall&scl=625000&R=1&uid=NULLGWDOCOMO
   ================================================== */

class Ktai_Location_iMapion extends Ktai_Location {

static public function factory($content) {
	if (! preg_match('#http://i\.mapion\.co\.jp/c/f\?(\S+)#m', $content, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_iMapion;
	$loc->url = $m[0];
	$params = parent::get_params($m[1]);
	$lat = str_replace('/', '.', @$params['nl']);
	$lon = str_replace('/', '.', @$params['el']);
	$loc->lat = parent::dms2deg($lat, 90);
	$loc->lon = parent::dms2deg($lon, 180);
	$loc->tokyo2wgs84();
	return $loc;
}

} // End of class

/* ==================================================
 *   Ktai_Location_iZenrin Class
 * example: http://i.i.zenrin.co.jp/MapToLink/p1?scd=00300&rl=%2fzi%2fmenu%2far1%3farea%3d07900&x=6&n=35.616297&e=139.565364&uid=NULLGWDOCOMO
   ================================================== */

class Ktai_Location_iZenrin extends Ktai_Location {

static public function factory($content) {
	if (! preg_match('#http://i\.i\.zenrin\.co\.jp/MapToLink/p1\?(\S+)#m', $content, $m)) {
		return NULL;
	}
	$loc = new Ktai_Location_iZenrin;
	$loc->url = $m[0];
	$params = parent::get_params($m[1]);
	$loc->lat = floatval(@$params['n']);
	$loc->lon = floatval(@$params['e']);
	$loc->tokyo2wgs84();
	return $loc;
}

} // End of class

?>