/* $Id: httpd.c,v 1.2 2003/01/20 00:05:03 rtakano Exp $
 * Simple HTTP server kernel
 * Copyright (c) 2002 TAKANO Ryousei
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <oskit/dev/dev.h>
#include <oskit/dev/osenv.h>
#include <oskit/dev/error.h>
#include <oskit/dev/linux.h>
#include <oskit/dev/freebsd.h>
#include <oskit/clientos.h>
#include <oskit/dev/net.h>
#include <oskit/dev/ethernet.h>
#include <oskit/net/freebsd.h>
#include <oskit/net/socket.h>
#include <oskit/x86/pc/base_multiboot.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "bootp.h"
#include "httpd.h"

static oskit_socket_factory_t* socket_factory;

#include "zipfile.h"
#include "ziphash.h"
static ZipFile *zip;

static int load_content(void)
{
  int i;
  struct multiboot_module* mod;
  ZipDirectory *zipd;

  multiboot_info_dump(&boot_info);

  if (boot_info.mods_count == 0) return 0;
  mod = (struct multiboot_module*)boot_info.mods_addr;

  if (!zip) {
    char* start;
    int size;
    start = (char*)mod->mod_start;
    size = (int)(mod->mod_end - mod->mod_start);
    zip = zip_archive_open(0, 0);
    zip_archive_assign(zip, start, size);
    if (read_zip_archive(zip) < 0) {
      return 0;
    }
  }

  zipd = (ZipDirectory*)zip->central_directory;
  for (i = 0; i < zip->count; i++, zipd = ZIPDIR_NEXT(zipd)) {
#if 0
    printf("%d: size:%ld, name(#%ld)%s, offset:%ld, csize:%ld, method:%ld\n",
	    i, zipd->size, zipd->filename_length,
	    ZIPDIR_FILENAME (zipd),
	    zipd->filestart,
	    zipd->compressed_size,
	    zipd->compression_method);
#endif
    zip_register_file(zipd);
  }
  return zip->count;
}

void http_do_request(oskit_stream_t* stream, const char* line)
{
  char file[256];
  char* p;
  char* tok;
  char* buf;
  oskit_u32_t size, read, written;
  oskit_error_t err;
  ZipDirectory* zipd;

  p = (char*)line;
  tok = strsep(&p, " \t");
  if (tok == 0 || strncmp(tok, "GET", 3)) {
    goto invalid;
  } 

  tok = strsep(&p, " \t");
  if (tok == 0) {
    goto invalid;
  }

  strcpy(file, "html");
  strcat(file, tok);
  if (file[strlen(file)-1] == '/') {
    strcat(file, "index.html");
  }
  printf("http_do_get: %s\n", file);

  if (!zip) goto invalid;

  zipd = zip_find_file(file);
  if (zipd == NULL) { /* File Not Found */
    err = oskit_stream_write(stream, HTTP_404, sizeof(HTTP_404)-1, &written);
    assert(!err);
    return;
  }

  read = zip_read_file(zip, zipd, &buf, &size);
  if (read != size) {
    goto invalid;
  }
  err = oskit_stream_write(stream, HTTP_200, sizeof(HTTP_200)-1, &written);
  assert(!err);
  err = oskit_stream_write(stream, buf, size, &written);
  assert(!err);
  return;

 invalid:
  err = oskit_stream_write(stream, HTTP_400, sizeof(HTTP_400)-1, &written);
  assert(!err);
}


static void http_server(void)
{
  oskit_socket_t* sock;
  oskit_socket_t* nsock;
  oskit_stream_t* stream;
  struct sockaddr_in saddr;
  oskit_u32_t salen;
  oskit_error_t err;

  if ((err = oskit_socket_factory_create(socket_factory,
		 OSKIT_AF_INET, OSKIT_SOCK_STREAM, IPPROTO_TCP, &sock))) {
    printf("oskit_socket_factory_create: errno = %d\n", err);
    return;
  }

  memset(&saddr, 0, salen = sizeof(saddr));
  saddr.sin_addr.s_addr = htonl(INADDR_ANY);
  saddr.sin_family = OSKIT_AF_INET;
  saddr.sin_port = htons(80);

  err = oskit_socket_bind(sock, (struct oskit_sockaddr*)&saddr, sizeof(saddr));
  assert(!err);

  err = oskit_socket_listen(sock, 5);
  assert(!err);

  while (1) {
    char buf[BUFSIZ];
    oskit_u32_t read;

    err = oskit_socket_accept(sock, (struct oskit_sockaddr*)&saddr, &salen, &nsock);
    assert(!err);

    /* get the stream interface */
    err = oskit_socket_query(nsock, &oskit_stream_iid, (void**)&stream);
    assert(!err);

    /* get HTTP request */
    err = oskit_stream_read(stream, buf, sizeof(buf), &read);
    assert(!err);

    http_do_request(stream, (const char*)buf);

    oskit_stream_release(stream);
    oskit_socket_release(nsock);
  }

  oskit_socket_release(sock);
}


int main(int argc, char** argv)
{
  struct oskit_freebsd_net_ether_if* eif;
  oskit_etherdev_t** etherdev;
  oskit_etherdev_t* dev;
  oskit_osenv_t* osenv;
  char ipaddr[20];
  char gateway[20];
  char netmask[20];
  int ndev;
  oskit_error_t err;

  oskit_clientos_init();

  load_content();

  printf("Initializing devices...\n");
  osenv = oskit_osenv_create_default();
  oskit_register(&oskit_osenv_iid, (void*)osenv);
  oskit_dev_init(osenv);
  oskit_linux_init_osenv(osenv);
  oskit_linux_init_net();

  printf("Probing devices...\n");
  oskit_dev_probe();
  oskit_dump_devices();

  /* initialize network code */
  err = oskit_freebsd_net_init(osenv, &socket_factory);
  assert(!err);

  /* find all the Ethernet device nodes, however we use the first one. */
  ndev = osenv_device_lookup(&oskit_etherdev_iid, (void***)&etherdev);
  if (ndev <= 0)
    panic("no Ethernet adaptors found!");

  dev = etherdev[0];

  /* get the IP address & other info */
  get_ipinfo(dev, ipaddr, gateway, netmask, 0);

  err = oskit_freebsd_net_open_ether_if(dev, &eif);
  assert(!err);

  /* configure the interface */
  err = oskit_freebsd_net_ifconfig(eif, "eth0", ipaddr, netmask);
  assert(!err);

  err = oskit_freebsd_net_add_default_route(gateway);
  assert(!err);

  printf("now starting service.\n");
  printf("type 'telnet %s 80' to test\n", ipaddr);
  http_server();

  /* close etherdev and release net_io devices */
  oskit_freebsd_net_close_ether_if(eif);
  return 0;
}
