<?php
/*
 * Copyright (c) 2008, YggDore Co., Ltd. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of YggDore Co., Ltd. nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Version 0.0.1
 */


/*
 * Configuration
 */

// GroupID
// Please set ID that you thought about.
// Alphabet and number, and the part of sign can be used.
// 32 charactors(bytes) format.
define( "YGG_GROUP_ID", "---- Set Group ID (32chars) ----" );

// String Encoding
// Please set your site encoding below.
// (Japan)
// Japanese(UTF-8) - 5001
// Japanese(EUC)   - 5002
define( "YGG_ENC_ID", 5001 );

// Time until time-out after last accesses
// Please set second time ater user access time it at end.
// If you set 0, then becomes invalid. 
define( "YGG_LATO_SEC", 3600 );

// Session name
// Please set client session name.
// If you set empty string, then be used default name in php.ini
define( "YGG_CS_NAME", "" );

// Login server address and base.(SSL)
// Please set login server address and base url.(SSL)
define( "YGG_LOGIN_SERVER", "www.yggdore.com" );
define( "YGG_LOGIN_BASE", "" );

// Login session server SSL connection mode.
// Please set SSL mode, when access to login session server.
// If you set TRUE, then be used SSL connection. (Need OpenSSL)
define( "YGG_LOGIN_SESSION_ISSSL", FALSE );

// Login session server address and base.
// Please set login session server address and base url.
define( "YGG_LOGIN_SESSION_SERVER", "www.yggdore.com" );
define( "YGG_LOGIN_SESSION_BASE", "" );

// User information saving array key name to $_SESSION.
define( "YGG_LOGIN_KEY", "YggDore" );



/*
 * YggDore setting.
 * Don't change below settings!
 */

// Request base key name
define( "YGG_BASE_KEY",	"ygg-" );

// Session connection request key name.
define( "YGG_CURL_KEY",	YGG_BASE_KEY . "curl" );

// Group ID request key name.
define( "YGG_GID_KEY",	YGG_BASE_KEY . "gid" );

// Request type (GET/POST) request key name.
define( "YGG_SRMD_KEY",	YGG_BASE_KEY . "rmd" );

// Request keeper request key name.
define( "YGG_SRDT_KEY",	YGG_BASE_KEY . "rdt" );

// Session load request key name.
define( "YGG_SID_KEY",	YGG_BASE_KEY . "sid" );

// String Encode for user information load request key name.
define( "YGG_ENC_KEY",	YGG_BASE_KEY . "enc" );



// YggDore session start function.
function yggdore_start( $config = NULL )
{
	try{
		return YggDore::start( $config );
	}
	catch( YggDoreRedirectException $e ){
		return NULL;
	}
	catch( RuntimeException $e ){
		YggDore::message( YggDore::MSG_BUSY );
		return NULL;
	}
	catch( Exception $e ){
		YggDore::message( YggDore::MSG_PROG );
		return NULL;
	}
}



// YggDore session end function.
function yggdore_end( $config = NULL )
{
	try{
		YggDore::end( $config );
	}
	catch( YggDoreRedirectException $e ){
		return;
	}
	catch( RuntimeException $e ){
		YggDore::message( YggDore::MSG_BUSY );
	}
	catch( Exception $e ){
		YggDore::message( YggDore::MSG_PROG );
	}
}



// YggDore test function.
function yggdore_test()
{
	// Check Group ID
	$gidm = "OK";
	if( !YggDoreTool::checkID( YGG_GROUP_ID ) ){
		$gidm = "NG (Check ID length (32 chars) and illegal charactor)";
	}

	echo '<html><body>';
	echo 'Group ID Check : ' . $gidm . '<br />';
	echo 'Current URL : ' . YggDoreTool::curURL() . '<br />';
	echo '</body></html>';
}



// YggDore oparation class
class YggDore {
	// Message status
	const MSG_PROG    = "prog";
	const MSG_BUSY    = "busy";
	const MSG_LOGOUT  = "lo";
	const MSG_TIMEOUT = "to";


	// YggDore session start method
	public static function start( $config = NULL )
	{
		// Session status check
		if( session_id() != "" ){
			throw new BadMethodCallException;
		}

		// Setting session name
		$cnm = YggDoreTool::getConfig( $config, "CSNAME", YGG_CS_NAME );
		if( $cnm != "" ){
			session_name( $cnm );
		}

		// Start session
		session_start();
		if( $_SESSION === NULL ){
			$_SESSION = array();
		}

		// Start YggDore session
		return self::startNSS( $config );
	}


	// YggDore session start method without session_start()
	public static function startNSS( $config = NULL )
	{
		// Session status check
		if( session_id() == "" ){
			throw new BadMethodCallException;
		}

		// Get Login key name
		$lk = YggDoreTool::getConfig( $config, "LKEY", YGG_LOGIN_KEY );

		// Load session data(user information and request keep data)
		if( isset($_REQUEST[YGG_SID_KEY]) ){
			// Load session data from login server
			$si = self::loadSession( $_REQUEST[YGG_SID_KEY], $config );
			$sd = $si->get();

			// Recover request data
			$req = new YggDoreRequest;
			$req->setReqKeep( $si );
			$_REQUEST = array_merge( $_REQUEST, $req->getData() );

			// Set session data
			$_SESSION[$lk]        = $sd['LOGIN'];
			$_SESSION[$lk]['LAT'] = time();

			return array(
				"REQUEST" => $req,
				"LOGIN"   => $sd['LOGIN']
			);
		}

		// Timeout judgement
		if( isset($_SESSION[$lk]) ){
			// Load to array key name of last access timeout
			$lato = YggDoreTool::getConfig(
				$config, "LATO", YGG_LATO_SEC
			);
			$nowt = time();
			if( $lato <= 0 || $nowt - $_SESSION[$lk]['LAT'] <= $lato ){
				// If don't timeout, update last access time
				$_SESSION[$lk]['LAT'] = $nowt;
				return array(
					"REQUEST" => $_REQUEST,
					"LOGIN"   => $_SESSION[$lk]
				);
			}

			// Timeout logout proc
			self::clear();

			// Redirect to YggDore Login
			self::login( self::MSG_TIMEOUT, $config );

			throw new YggDoreRedirectException;
		}

		// Redirect to YggDore Login
		self::login( "", $config );

		throw new YggDoreRedirectException;
	}


	// YggDore session end method
	public static function end( $config = NULL )
	{
		// Session status check
		if( session_id() != "" ){
			throw new BadMethodCallException;
		}

		// Setting session name
		$cnm = YggDoreTool::getConfig( $config, "CSNAME", YGG_CS_NAME );
		if( $cnm != "" ){
			session_name( $cnm );
		}

		// Start session
		session_start();

		// Delete YggDore session
		self::endNSS( $config );
	}


	// YggDore session end method without session_start()
	public static function endNSS( $config = NULL )
	{
		// Clear session data
		self::clear();

		// Redirect to YggDore Login
		self::login( self::MSG_LOGOUT, $config );

		throw new YggDoreRedirectException;
	}


	// Clear YggDore session data
	public static function clear()
	{
		// Session status check
		if( session_id() == "" ){
			throw new BadMethodCallException;
		}

		// Delete all session data
		$_SESSION = array();

		// Delete session cookie
		$cnm = session_name();
		if( isset($_COOKIE[$cnm]) ){
			setcookie( $cnm, "", time() - 42000, "/" );
		}

		// Delete session
		session_destroy();
	}


	// Save session (before login process) data
	public static function saveSession( $data = NULL, $config = NULL )
	{
		// Load group ID request name
		$gid = YggDoreTool::getConfig( $config, "GID", YGG_GROUP_ID );

		// Load user info string encode
		$enc = YggDoreTool::getConfig( $config, "ENC", YGG_ENC_ID );

		// Get current site URL
		$curl = YggDoreTool::getConfig(
			$config, "CURL", YggDoreTool::curURL()
		);

		// Save session data
		return YggDoreSession::save( $gid, $curl, $enc, $data );
	}


	// Load session (after login process) data
	public static function loadSession( $sid, $config = NULL )
	{
		// Load group ID request name
		$gid = YggDoreTool::getConfig( $config, "GID", YGG_GROUP_ID );

		// Load session data
		$data = new YggDoreSessionInfo;
		$data->set( $gid, $sid );

		return $data;
	}


	// Get redirect header to login
	public static function getLoginURL( $sid, $msg = "" )
	{
		// Check argument
		if( $sid == "" ){
			throw new UnexpectedValueException;
		}

		// Logout proc request
		if( $msg != "" ){
			$msg = "&ygg-msg=" . urlencode( $msg );
		}

		// Create redirect to login URL
		return "Location: " .
		       "http://" .
			   YGG_LOGIN_SERVER .
			   YGG_LOGIN_BASE .
			   "/l/?ygg-sid=" . urlencode( $sid ) . $msg;
	}


	// Login process
	public static function login( $msg = "", $config = NULL )
	{
		// Get web request data
		$req = new YggDoreRequest;

		// Save session data
		$sid = self::saveSession( $req, $config );

		// Redirect to Login
		header( self::getLoginURL( $sid, $msg ) );
	}


	// Get redirect header to message
	public static function getMessageURL( $msg = self::PROG )
	{
		// Create redirect to error URL
		return "Location: " .
		       "http://" .
			   YGG_LOGIN_SERVER .
			   YGG_LOGIN_BASE .
			   "/l/m/?ygg-msg=" . urlencode( $msg );
	}


	// Message process
	public static function message( $msg = self::PROG )
	{
		// Redirect to Login
		header( self::getMessageURL( $sid ) );
	}
}



// Redirect Exception
class YggDoreRedirectException extends RuntimeException {
}



// Session data interface
interface YggDoreSessionSaveInterface {
	public function getSessionData();
}



// Web request data class
class YggDoreRequest implements YggDoreSessionSaveInterface {
	// Request method mode
	const GET  = 1;
	const POST = 2;


	// Class member
	private $_mode;
	private $_data;


	// Constructor
	public function __construct()
	{
		// Get web request data
		if( strtoupper($_SERVER["REQUEST_METHOD"]) == "GET" ){
			$this->_mode = self::GET;
			$this->_data = $_GET;
		}
		else{
			$this->_mode = self::POST;
			$this->_data = $_POST;
		}
	}


	// Get request method mode
	public function getMode()
	{
		return $this->_mode;
	}


	// Get request data
	public function getData()
	{
		return $this->_data;
	}


	// Get all member data of instance
	public function get()
	{
		return array(
			"MODE" => $this->_mode,
			"DATA" => $this->_data
		);
	}


	// Get YggDore format session data
	public function getSessionData()
	{
		return array(
			YGG_SRMD_KEY => $this->_mode,
			YGG_SRDT_KEY => serialize( $this->_data )
		);
	}


	// Set from YggDore format web request data
	public function setReqKeep( $si )
	{
		// Check argument
		$rk = $si->getReqKeep();
		if( !isset( $rk[YGG_SRMD_KEY] ) ){
			throw new UnexpectedValueException;
		}
		elseif( !isset( $rk[YGG_SRDT_KEY] ) ){
			throw new UnexpectedValueException;
		}
		elseif( $rk[YGG_SRDT_KEY] == "" ){
			throw new UnexpectedValueException;
		}

		// Check web request mode
		if( $rk[YGG_SRMD_KEY] != (string)self::GET &&
			$rk[YGG_SRMD_KEY] != (string)self::POST ){
			throw new UnexpectedValueException;
		}

		// Get web request data
		$rk[YGG_SRDT_KEY] = unserialize( $rk[YGG_SRDT_KEY] );
		if( $rk[YGG_SRDT_KEY] === FALSE ){
			throw new RuntimeException;
		}

		// Set to member
		$this->_mode = (int)$rk[YGG_SRMD_KEY];
		$this->_data = $rk[YGG_SRDT_KEY];
	}
}



// Session class
class YggDoreSession {
	// Save session data
	public static function save(
		$gid, $curl, $enc, $data = NULL, $config = NULL
	)
	{
		// Check argument
		if( !YggDoreTool::checkID( $gid ) ){
			throw new UnexpectedValueException;
		}
		elseif( !preg_match( '/^http[s]{0,1}:\/\/[[:graph:]]+$/', $curl ) ){
			throw new UnexpectedValueException;
		}
		elseif( $enc <= 0 ){
			throw new UnexpectedValueException;
		}

		// Create connection status
		$pno = 80;
		if( YGG_LOGIN_SESSION_ISSSL ){
			$pno = 443;
		}

		// Create request data
		$post = array(
			YGG_GID_KEY  => $gid,
			YGG_CURL_KEY => $curl,
			YGG_ENC_KEY  => $enc
		);
		if( $data !== NULL ){
			$post = array_merge( $data->getSessionData(), $post );
		}

		// Send loading session data request
		$xmle = YggDoreTool::reqPostXML(
			YGG_LOGIN_SESSION_SERVER,
			$pno,
			YGG_LOGIN_SESSION_BASE . "/s/n/",
			$post,
			YGG_LOGIN_SESSION_ISSSL
		);

		// Parse data
		if( $xmle->getName() != "SessionSaveResponce" ){
			throw new RuntimeException;
		}
		elseif( $xmle->ID == "" ){
			throw new RuntimeException;
		}

		return (string)$xmle->ID;
	}
}



// Load session data class
class YggDoreSessionInfo {
	// Member
	private $_gid;
	private $_sid;
	private $_li;
	private $_rk;


	// Constructor
	public function __construct()
	{
		$this->_gid = NULL;
		$this->_sid = NULL;
		$this->_li  = NULL;
		$this->_rk  = NULL;
	}


	// Get group ID
	public function getGID()
	{
		if( $this->_gid === NULL ){
			throw new BadMethodCallException;
		}

		return $this->_gid;
	}


	// Get session ID
	public function getSID()
	{
		if( $this->_gid === NULL ){
			throw new BadMethodCallException;
		}

		return $this->_sid;
	}


	// Get user information
	public function getLogin()
	{
		if( $this->_gid === NULL ){
			throw new BadMethodCallException;
		}

		return $this->_li;
	}


	// Get web request keep data
	public function getReqKeep()
	{
		if( $this->_gid === NULL ){
			throw new BadMethodCallException;
		}

		return $this->_rk;
	}


	// Get all member data of instance
	public function get()
	{
		if( $this->_gid === NULL ){
			throw new BadMethodCallException;
		}

		return array(
			"GID"   => $this->_gid,
			"SID"   => $this->_sid,
			"LOGIN" => $this->_li,
			"RK"    => $this->_rk
		);
	}


	// Load session process
	public function set( $gid, $sid, $config = NULL )
	{
		// Check argument
		if( !YggDoreTool::checkID( $gid ) ){
			throw new UnexpectedValueException;
		}
		elseif( !YggDoreTool::checkID( $sid ) ){
			throw new UnexpectedValueException;
		}

		// Create port no
		$pno = 80;
		if( YGG_LOGIN_SESSION_ISSSL ){
			$pno = 443;
		}

		// Send saving session data request
		$xmle = YggDoreTool::reqPostXML(
			YGG_LOGIN_SESSION_SERVER,
			$pno,
			YGG_LOGIN_SESSION_BASE . "/s/l/",
			array(
				YGG_GID_KEY => $gid,
				YGG_SID_KEY => $sid
			),
			YGG_LOGIN_SESSION_ISSSL
		);

		// Parse data
		if( $xmle->getName() != "SessionInfoResponce" ){
			throw new RuntimeException;
		}

		// Analy user information data
		$li = array(
			"ID"     => (string)$xmle->LoginInfo->ID,
			"NNAME"  => (string)$xmle->LoginInfo->Nickname,
			"NNAMEL" => base64_decode(
				(string)$xmle->LoginInfo->LocalNickname
			),
			"DOMAIN" => (string)$xmle->LoginInfo->Domain,
			"RT"     => (int)$xmle->LoginInfo->RegistTimestamp
		);

		// Analy web request keep data
		$rk = array();
		foreach( $xmle->RequestKeeper->children() as $cxmle ){
			$name      = base64_decode( (string)$cxmle->Name );
			$rk[$name] = base64_decode( (string)$cxmle->Value );
		}

		// Set member
		$this->_gid = $gid;
		$this->_sid = $sid;
		$this->_li  = $li;
		$this->_rk  = $rk;
	}
}


// Common function tools class
class YggDoreTool {
	// Send POST request
	public static function reqPost(
		$host, $port, $path, $data, $isssl = TRUE
	)
	{
		// Check argument
		if( $host == "" ){
			throw new UnexpectedValueException;
		}
		elseif( $port <= 0 ){
			throw new UnexpectedValueException;
		}
		elseif( $path == "" ){
			throw new UnexpectedValueException;
		}
		elseif( count($data) <= 0 ){
			throw new UnexpectedValueException;
		}

		// Create sending web request data
		$data = http_build_query( $data );
		if( $data == "" ){
			throw new UnexpectedValueException;
		}

		// SSL mode fix
		$sslmd = "";
		if( $isssl ){
			$sslmd = "ssl://";
		}

		// Open stream
		$fp = fsockopen( $sslmd . $host, $port, $errno, $errstr, 90 );
		if( !$fp ){
			throw new RuntimeException;
		}

		try{
			// Setting stream blocking mode
			$rts = stream_set_blocking( $fp, TRUE );
			if( !$rts ){
				throw new RuntimeException;
			}

			// Setting stream timeout
			$rts = stream_set_timeout( $fp, 90 );
			if( !$rts ){
				throw new RuntimeException;
			}

			// Send web request data
			$sndh = "POST " . $path . " HTTP/1.0\r\n" .
			        "HOST: " . $host . "\r\n" .
					"User-Agent: YGGClient\r\n" .
					"Content-Type: application/x-www-form-urlencoded\r\n" .
					"Content-Length: " . strlen( $data ) . "\r\n" .
					"\r\n";
			$rts = fwrite( $fp, $sndh );
			if( !$rts ){
				throw new RuntimeException;
			}
			$rts = fwrite( $fp, $data );
			if( !$rts ){
				throw new RuntimeException;
			}

			// Read request response data
			$line = stream_get_line( $fp, 8192, "\r\n" );
			if( $line === FALSE ){
				throw new RuntimeException;
			}
			$info = stream_get_meta_data( $fp );
			if( !$info || $info['timed_out'] ){
				// Check stream timeout
				throw new RuntimeException;
			}
			elseif( !preg_match( '/^HTTP\/[0-9\.]+[\s]+200/', $line ) ){
				// Check HTTP response
				throw new RuntimeException;
			}

			// Read response header
			while( !feof( $fp ) ){
				// Read data per line
				$line = stream_get_line( $fp, 8192, "\r\n" );
				if( $line === FALSE ){
					throw new RuntimeException;
				}

				// Check stream timeout
				$info = stream_get_meta_data( $fp );
				if( !$info || $info['timed_out'] ){
					throw new RuntimeException;
				}

				// Loop break when response data is empty
				if( $line == "" ){
					break;
				}
			}

			// Load data
			$rdata = stream_get_contents( $fp );
			if( $rdata === FALSE ){
				throw new RuntimeException;
			}

			// Check stream timeout
			$info = stream_get_meta_data( $fp );
			if( !$info || $info['timed_out'] ){
				throw new RuntimeException;
			}
		}
		catch( Exception $e ){
			fclose( $fp );
			throw $e;
		}

		// Close stream
		fclose( $fp );

		return $rdata;
	}


	// Send POST request and parse XML response
	public static function reqPostXML(
		$host, $port, $path, $data = NULL, $isssl = TRUE
	)
	{
		// Send POST request
		$data = self::reqPost(
			$host, $port, $path, $data, $isssl
		);

		// Parse XML response
		return new SimpleXMLElement( $data );
	}


	// Create current site URL
	public static function curURL()
	{
		$curl = "http://";
		if( isset($_SERVER['HTTPS']) &&
			strtoupper($_SERVER['HTTPS']) == "ON" ){
			$curl = "https://";
		}
		return $curl .
		       $_SERVER["HTTP_X_FORWARDED_HOST"] .
			   $_SERVER["PHP_SELF"];
	}


	// Load configuration data
	public static function getConfig( $config, $key, $default = NULL )
	{
		if( !isset( $config[$key] ) ){;
			return $default;
		}
		return $config[$key];
	}


	// Check ID
	public static function checkID( $id )
	{
		if( strlen( $id ) != 32 ){
			return FALSE;
		}
		elseif( !preg_match( '/^[[:graph:]]+$/', $id ) ){
			return FALSE;
		}

		return TRUE;
	}
}
?>
