
class InstallError < StandardError; end


class Installer

  Version   = '3.0.2'
  Copyright = 'Copyright (c) 2000,2001 Minero Aoki'


  @toplevel = nil

  def self.declear_toplevel_installer( inst )
    @toplevel and
        raise ArgumentError, 'more than one toplevel installer decleared'
    @toplevel = inst
  end

  def self.toplevel_installer
    @toplevel
  end


  FILETYPES = %w( bin lib ext data )

  include FileOperations

  def initialize( config, opt, srcroot, objroot )
    @config = config
    @options = opt
    @srcdir = File.expand_path(srcroot)
    @objdir = File.expand_path(objroot)
    @currdir = '.'
  end

  def inspect
    "#<#{type} #{__id__}>"
  end

  #
  # configs/options
  #

  def config( key )
    @config[key]
  end

  def no_harm?
    @options['no-harm']
  end

  def verbose?
    @options['verbose']
  end

  def verbose_off
    save, @options['verbose'] = @options['verbose'], false
    yield
    @options['verbose'] = save
  end

  #
  # srcdir/objdir
  #

  attr_reader :srcdir
  alias srcdir_root srcdir
  alias package_root srcdir

  def curr_srcdir
    "#{@srcdir}/#{@currdir}"
  end

  attr_reader :objdir
  alias objdir_root objdir

  def curr_objdir
    "#{@objdir}/#{@currdir}"
  end

  def srcfile( path )
    curr_srcdir + '/' + path
  end

  def srcexist?( path )
    File.exist? srcfile(path)
  end

  def srcdirectory?( path )
    dir? srcfile(path)
  end
  
  def srcfile?( path )
    File.file? srcfile(path)
  end

  def srcentries( path = '.' )
    Dir.open( curr_srcdir + '/' + path ) {|d|
        return d.to_a - %w(. ..) - hookfilenames
    }
  end

  def srcfiles( path = '.' )
    srcentries(path).find_all {|fname|
        File.file? File.join(curr_srcdir, path, fname)
    }
  end

  def srcdirectories( path = '.' )
    srcentries(path).find_all {|fname|
        dir? File.join(curr_srcdir, path, fname)
    }
  end

  def dive_into( rel )
    return unless dir? "#{@srcdir}/#{rel}"

    dir = File.basename(rel)
    Dir.mkdir dir unless dir? dir
    save = Dir.pwd
    Dir.chdir dir
    $stderr.puts '---> ' + rel if verbose?
    @currdir = rel
    yield
    Dir.chdir save
    $stderr.puts '<--- ' + rel if verbose?
    @currdir = File.dirname(rel)
  end

  #
  # config
  #

  def exec_config
    exec_task_traverse 'config'
  end

  def config_dir_bin( rel )
  end

  def config_dir_lib( rel )
  end

  def config_dir_ext( rel )
    extconf if extdir? curr_srcdir
  end

  def extconf
    opt = @options['config-opt'].join(' ')
    command "#{config('ruby-prog')} #{curr_srcdir}/extconf.rb #{opt}"
  end

  def config_dir_data( rel )
  end

  #
  # setup
  #

  def exec_setup
    exec_task_traverse 'setup'
  end

  def setup_dir_bin( relpath )
    all_files( curr_srcdir ).each do |fname|
      add_rubypath "#{curr_srcdir}/#{fname}"
    end
  end

  SHEBANG_RE = /\A\#!\s*\S*ruby\S*/

  def add_rubypath( path )
    $stderr.puts %Q<set #! line to "\#!#{config('ruby-path')}" for #{path} ...> if verbose?
    return if no_harm?

    tmpfile = File.basename(path) + '.tmp'
    begin
      File.open( path ) {|r|
      File.open( tmpfile, 'w' ) {|w|
        first = r.gets
        return unless SHEBANG_RE === first   # reject '/usr/bin/env ruby'

        w.print first.sub( SHEBANG_RE, '#!' + config('ruby-path') )
        w.write r.read
      } }
      mv tmpfile, File.basename(path)
    ensure
      rm_f tmpfile if File.exist? tmpfile
    end
  end

  def setup_dir_lib( relpath )
  end

  def setup_dir_ext( relpath )
    if extdir? curr_srcdir then
      make
    end
  end

  def make
    command config('make-prog')
  end

  def setup_dir_data( relpath )
  end

  #
  # install
  #

  def exec_install
    exec_task_traverse 'install'
  end

  def install_dir_bin( rel )
    install_files targfiles, config('bin-dir') + '/' + rel, 0755
  end

  def install_dir_lib( rel )
    install_files targfiles, config('rb-dir') + '/' + rel, 0644
  end

  def install_dir_ext( rel )
    if extdir? curr_srcdir then
      install_dir_ext_main File.dirname(rel)
    end
  end

  def install_dir_ext_main( rel )
    install_files allext('.'), config('so-dir') + '/' + rel, 0555
  end

  def install_dir_data( rel )
    install_files targfiles, config('data-dir') + '/' + rel, 0644
  end

  def install_files( list, dest, mode )
    mkdir_p dest
    list.each do |fname|
      install fname, dest, mode
    end
  end
  
  def targfiles
    (targfilenames - hookfilenames).collect {|fname|
        File.exist?(fname) ? fname : File.join(curr_srcdir, fname)
    }
  end

  def targfilenames
    [ curr_srcdir, '.' ].inject([]) {|ret, dir|
        ret | all_files(dir)
    }
  end

  def hookfilenames
    %w( pre-%s post-%s pre-%s.rb post-%s.rb ).collect {|fmt|
        %w( config setup install clean ).collect {|t| sprintf fmt, t }
    }.flatten
  end

  def allext( dir )
    _allext(dir) or raise InstallError,
        "no extention exists: Have you done 'ruby #{$0} setup' ?"
  end

  DLEXT = /\.#{ ::Config::CONFIG['DLEXT'] }\z/

  def _allext( dir )
    Dir.open( dir ) {|d|
      return d.find_all {|fname| DLEXT === fname }
    }
  end

  #
  # clean
  #

  def exec_clean
    exec_task_traverse 'clean'
    rm_f 'config.save'
    rm_f 'InstalledFiles'
  end

  def clean_dir_bin( rel )
  end

  def clean_dir_lib( rel )
  end

  def clean_dir_ext( rel )
    clean
  end
  
  def clean
    command config('make-prog') + ' clean' if File.file? 'Makefile'
  end

  def clean_dir_data( rel )
  end

  #
  # lib
  #

  def exec_task_traverse( task )
    run_hook 'pre-' + task
    FILETYPES.each do |type|
      traverse task, type, task + '_dir_' + type
    end
    run_hook 'post-' + task
  end

  def traverse( task, rel, mid )
    dive_into( rel ) {
      run_hook 'pre-' + task
      __send__ mid, rel.sub( %r_\A.*?(?:/|\z)_, '' )
      all_dirs( curr_srcdir ).each do |d|
        traverse task, rel + '/' + d, mid
      end
      run_hook 'post-' + task
    }
  end

  def run_hook( name )
    try_run_hook curr_srcdir + '/' + name           or
    try_run_hook curr_srcdir + '/' + name + '.rb'
  end

  def try_run_hook( fname )
    return false unless File.file? fname

    env = self.dup
    s = nil
    File.open( fname ) {|f| s = f.read }
    begin
      env.instance_eval s, fname, 1
    rescue
      raise InstallError, "hook #{fname} failed:\n" + $!.message
    end
    true
  end

  def extdir?( dir )
    File.exist? dir + '/MANIFEST'
  end

end

